<?php
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Admin_Columns {
    public static function register() {
        add_filter('manage_event_posts_columns', array(__CLASS__, 'add_columns'));
        add_action('manage_event_posts_custom_column', array(__CLASS__, 'manage_columns'), 10, 2);
        add_filter('manage_edit-event_sortable_columns', array(__CLASS__, 'make_columns_sortable'));
        add_action('pre_get_posts', array(__CLASS__, 'handle_orderby'));
        add_action('pre_get_posts', array(__CLASS__, 'set_default_sort'));
        
        // Taxonomy term list columns
        add_filter('manage_edit-artist_columns', array(__CLASS__, 'add_term_columns'));
        add_filter('manage_artist_custom_column', array(__CLASS__, 'manage_term_columns'), 10, 3);
        add_filter('manage_edit-tour_columns', array(__CLASS__, 'add_term_columns'));
        add_filter('manage_tour_custom_column', array(__CLASS__, 'manage_term_columns'), 10, 3);
        
        // Add Quick Edit support
        add_action('quick_edit_custom_box', array(__CLASS__, 'add_quick_edit_fields'), 10, 2);
        add_action('bulk_edit_custom_box', array(__CLASS__, 'add_bulk_edit_fields'), 10, 2);
        add_action('admin_enqueue_scripts', array(__CLASS__, 'load_quick_edit_scripts'));
        add_action('save_post_event', array(__CLASS__, 'save_quick_edit_data'), 10, 2);
        
        // Set event_date as the primary column
        add_filter('list_table_primary_column', array(__CLASS__, 'set_primary_column'), 10, 2);
    }
    
    public static function add_columns($columns) {
        $new_columns = array();
        
        // Get display format preference
        $display_format = Vizou_Events_Settings::get_option('display_format');
        $venue_first = ($display_format !== 'artist_first');
        
        foreach($columns as $key => $value) {
            if ($key === 'title') {
                // Date is always first
                $new_columns['event_date'] = esc_html__('Event Date', 'vizou-events-manager');
                
                // Then venue or artist based on setting
                if ($venue_first) {
                    $new_columns['title'] = esc_html__('Venue', 'vizou-events-manager');
                    $new_columns['artist'] = esc_html__('Artist', 'vizou-events-manager');
                } else {
                    $new_columns['artist'] = esc_html__('Artist', 'vizou-events-manager');
                    $new_columns['title'] = esc_html__('Venue', 'vizou-events-manager');
                }
                
                // Then the rest
                $new_columns['tour'] = esc_html__('Tour', 'vizou-events-manager');
                $new_columns['status'] = esc_html__('Status', 'vizou-events-manager');
                $new_columns['thumb'] = esc_html__('Thumb', 'vizou-events-manager');
            } else if ($key !== 'title') { 
                $new_columns[$key] = $value;
            }
        }
        
        return $new_columns;
    }

    // Set default event sorting for All Events
    public static function set_default_sort($query) {
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }

        // Only modify event post type listing
        $screen = get_current_screen();
        if (!$screen || $screen->post_type !== 'event') {
            return;
        }

        // Check if custom ordering is already set via query object instead of $_GET
        $current_orderby = $query->get('orderby');
        if (empty($current_orderby)) {
            $query->set('meta_key', 'event_start');
            $query->set('meta_type', 'DATE');
            $query->set('orderby', 'meta_value');
            $query->set('order', 'DESC');
        }
    }
    
    public static function manage_columns($column, $post_id) {
        switch ($column) {
            case 'event_date':
                $date = get_field('event_start', $post_id);
                if ($date) {
                    $date_obj = DateTime::createFromFormat('d/m/Y', $date);
                    if ($date_obj) {
                        echo esc_html(date_i18n(get_option('date_format'), $date_obj->getTimestamp()));
                    }
                }
                break;
                
            case 'title':
                $venues = get_the_terms($post_id, 'venue');
                if ($venues && !is_wp_error($venues)) {
                    $venue_links = array();
                    foreach ($venues as $venue) {
                        $venue_links[] = sprintf(
                            '<a href="%s">%s</a>',
                            esc_url(get_edit_term_link($venue->term_id, 'venue')),
                            esc_html($venue->name)
                        );
                    }
                    echo wp_kses_post(implode(', ', $venue_links));
                }
                break;
                
            case 'artist':
                $artists = get_the_terms($post_id, 'artist');
                if ($artists && !is_wp_error($artists)) {
                    $artist_links = array();
                    foreach ($artists as $artist) {
                        $artist_links[] = sprintf(
                            '<a href="%s">%s</a>',
                            esc_url(get_edit_term_link($artist->term_id, 'artist')),
                            esc_html($artist->name)
                        );
                    }
                    echo wp_kses_post(implode(', ', $artist_links));
                }
                break;

            case 'tour':
                $tours = get_the_terms($post_id, 'tour');
                if ($tours && !is_wp_error($tours)) {
                    $tour_links = array();
                    foreach ($tours as $tour) {
                        $tour_links[] = sprintf(
                            '<a href="%s">%s</a>',
                            esc_url(get_edit_term_link($tour->term_id, 'tour')),
                            esc_html($tour->name)
                        );
                    }
                    echo wp_kses_post(implode(', ', $tour_links));
                }
                break;
                
            case 'status':
                $status = get_field('event_status', $post_id);
                if ($status) {
                    echo esc_html(Vizou_Events_Settings::get_status_label($status));
                    
                    // Add hidden data for quick edit
                    echo '<div class="hidden" id="inline_' . esc_attr($post_id) . '">';
                    echo '<div class="event_status">' . esc_html($status) . '</div>';
                    
                    // Add tour data for quick edit
                    $tours = get_the_terms($post_id, 'tour');
                    $tour_id = '';
                    if ($tours && !is_wp_error($tours)) {
                        $tour_id = $tours[0]->term_id;
                    }
                    echo '<div class="event_tour">' . esc_html($tour_id) . '</div>';
                    echo '</div>';
                } else {
                    echo '—';
                }
                break;

            case 'thumb':
                // Only show thumbnail if Featured Image Admin Thumb plugin is not active
                if (!class_exists('Featured_Image_Admin_Thumb') && has_post_thumbnail($post_id)) {
                    $thumb = get_the_post_thumbnail($post_id, array(50, 50), array(
                        'alt' => esc_attr(get_the_title($post_id))
                    ));
                    echo wp_kses_post($thumb);
                }
                break;
        }
    }
    
    public static function make_columns_sortable($columns) {
        $columns['event_date'] = 'event_start';
        $columns['title'] = 'title';
        $columns['status'] = 'event_status';
        return $columns;
    }
    
    public static function handle_orderby($query) {
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }

        $orderby = $query->get('orderby');
        
        if ($orderby === 'event_start') {
            $query->set('meta_key', 'event_start');
            $query->set('meta_type', 'DATE');
            $query->set('orderby', 'meta_value');
        } elseif ($orderby === 'event_status') {
            $query->set('meta_key', 'event_status');
            $query->set('orderby', 'meta_value');
        }
    }
    
    public static function add_term_columns($columns) {
        $new_columns = array();
        foreach($columns as $key => $value) {
            $new_columns[$key] = $value;
        }
        return $columns;
    }
    
    public static function manage_term_columns($string, $column_name, $term_id) {
        return $string;
    }
    
    // Add Quick Edit fields for Event Status and Tour
    public static function add_quick_edit_fields($column_name, $post_type) {
        if ($post_type !== 'event' || $column_name !== 'status') {
            return;
        }
        
        $tours = get_terms(array(
            'taxonomy' => 'tour',
            'hide_empty' => false,
        ));
        ?>
        <fieldset class="inline-edit-col-right">
            <div class="inline-edit-col">
                <label>
                    <span class="title"><?php esc_html_e('Event status', 'vizou-events-manager'); ?></span>
                    <span class="input-text-wrap">
                        <select name="event_status">
                            <option value="Active"><?php echo esc_html(Vizou_Events_Settings::get_status_label('active')); ?></option>
                            <option value="Cancelled"><?php echo esc_html(Vizou_Events_Settings::get_status_label('cancelled')); ?></option>
                            <option value="Sold out"><?php echo esc_html(Vizou_Events_Settings::get_status_label('sold_out')); ?></option>
                            <option value="To be confirmed"><?php echo esc_html(Vizou_Events_Settings::get_status_label('reschedule')); ?></option>
                        </select>
                    </span>
                </label>
                
                <label>
                    <span class="title"><?php esc_html_e('Tour', 'vizou-events-manager'); ?></span>
                    <span class="input-text-wrap">
                        <select name="event_tour">
                            <option value="">— <?php esc_html_e('No change', 'vizou-events-manager'); ?> —</option>
                            <option value="0">— <?php esc_html_e('None', 'vizou-events-manager'); ?> —</option>
                            <?php if (!empty($tours) && !is_wp_error($tours)): ?>
                                <?php foreach ($tours as $tour): ?>
                                    <option value="<?php echo esc_attr($tour->term_id); ?>">
                                        <?php echo esc_html($tour->name); ?>
                                    </option>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </select>
                    </span>
                </label>
            </div>
        </fieldset>
        <?php
    }
    
    // Add Bulk Edit fields for Tour
    public static function add_bulk_edit_fields($column_name, $post_type) {
        if ($post_type !== 'event' || $column_name !== 'status') {
            return;
        }
        
        $tours = get_terms(array(
            'taxonomy' => 'tour',
            'hide_empty' => false,
        ));
        ?>
        <fieldset class="inline-edit-col-right">
            <div class="inline-edit-col">
                <label>
                    <span class="title"><?php esc_html_e('Tour', 'vizou-events-manager'); ?></span>
                    <span class="input-text-wrap">
                        <select name="event_tour">
                            <option value="">— <?php esc_html_e('No change', 'vizou-events-manager'); ?> —</option>
                            <option value="0">— <?php esc_html_e('None', 'vizou-events-manager'); ?> —</option>
                            <?php if (!empty($tours) && !is_wp_error($tours)): ?>
                                <?php foreach ($tours as $tour): ?>
                                    <option value="<?php echo esc_attr($tour->term_id); ?>">
                                        <?php echo esc_html($tour->name); ?>
                                    </option>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </select>
                    </span>
                </label>
            </div>
        </fieldset>
        <?php
    }
    
    // Load JavaScript for Quick Edit
    public static function load_quick_edit_scripts($hook) {
        if ('edit.php' !== $hook) {
            return;
        }
        
        $screen = get_current_screen();
        if (!$screen || 'event' !== $screen->post_type) {
            return;
        }
        
        wp_enqueue_script(
            'vizou-events-quick-edit',
            plugins_url('assets/js/quick-edit.js', dirname(__FILE__)),
            array('jquery', 'inline-edit-post'),
            '0.1.0',
            true
        );
    }
    
    // Save Quick Edit data - status and tour
    public static function save_quick_edit_data($post_id, $post) {
        // Skip autosaves
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check post type
        if ('event' != $post->post_type) {
            return;
        }
        
        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Verify nonce for quick edit
        if (isset($_POST['_inline_edit']) && isset($_POST['_wpnonce'])) {
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'inlineeditnonce')) {
                return;
            }
        }
        
        // Verify nonce for bulk edit  
        if (isset($_GET['bulk_edit']) && isset($_GET['_wpnonce'])) {
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'bulk-posts')) {
                return;
            }
        }
        
        // Now safely handle the form data
        $event_status = '';
        $event_tour = '';
        
        if (isset($_POST['event_status'])) {
            $event_status = sanitize_text_field(wp_unslash($_POST['event_status']));
        } elseif (isset($_GET['event_status'])) {
            $event_status = sanitize_text_field(wp_unslash($_GET['event_status']));
        }
        
        if (isset($_POST['event_tour'])) {
            $event_tour = sanitize_text_field(wp_unslash($_POST['event_tour']));
        } elseif (isset($_GET['event_tour'])) {
            $event_tour = sanitize_text_field(wp_unslash($_GET['event_tour']));
        }
        
        // Update event status if provided
        if (!empty($event_status)) {
            update_field('event_status', $event_status, $post_id);
        }
        
        // Update tour if provided
        if (isset($_POST['event_tour']) || isset($_GET['event_tour'])) {
            if ($event_tour === '0') {
                // Remove tour
                wp_set_object_terms($post_id, array(), 'tour');
            } elseif (!empty($event_tour) && is_numeric($event_tour)) {
                // Set tour
                wp_set_object_terms($post_id, array(intval($event_tour)), 'tour');
            }
            // If empty string, don't change (bulk edit "no change" option)
        }
    }
    
    public static function set_primary_column($column, $screen_id) {
        if ($screen_id === 'edit-event') {
            return 'event_date';
        }
        return $column;
    }
}

add_action('admin_init', array('Vizou_Events_Admin_Columns', 'register'));