<?php
/**
 * Handle artist creation and management
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Artist_Handler {
    public static function register() {
        // Handle artist field attributes
        add_filter('acf/prepare_field', array(__CLASS__, 'add_artist_field_attributes'));
        
        // Add AJAX handlers
        add_action('wp_ajax_get_artist_details', array(__CLASS__, 'get_artist_details'));
        
        // Add scripts for artist handling
        add_action('acf/input/admin_head', array(__CLASS__, 'add_immediate_load_script'));
        
        // Update artist field choices
        add_filter('acf/load_field/name=event_artist', array(__CLASS__, 'update_artist_field_choices'));
    }
    
    /**
     * Add data attributes to artist-related fields
     */
    public static function add_artist_field_attributes($field) {
        if ($field['name'] === 'event_artist_url') {
            $field['wrapper']['data-artist-field'] = 'url';
        }
        return $field;
    }
    
    /**
     * AJAX handler for getting artist details
     */
    public static function get_artist_details() {
        // Check user permissions
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Unauthorized access');
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(
            sanitize_text_field(wp_unslash($_POST['nonce'])), 
            'get_artist_details'
        )) {
            wp_send_json_error('Invalid security token');
        }

        // Validate term_id
        if (!isset($_POST['term_id'])) {
            wp_send_json_error('No term ID provided');
        }

        $term_id = absint($_POST['term_id']);
        
        // Verify term exists and is the correct taxonomy
        $term = get_term($term_id, 'artist');
        if (is_wp_error($term) || !$term) {
            wp_send_json_error('Invalid artist');
        }
        
        // Get URL from term meta
        $url = get_term_meta($term_id, 'url', true);
        
        wp_send_json_success(array(
            'url' => $url ? esc_url($url) : ''
        ));
    }

    /**
     * Add script for immediate loading of artist details
     */
    public static function add_immediate_load_script() {
        global $post_type;
        if ($post_type !== 'event') {
            return;
        }
        ?>
        <script type="text/javascript">
        (function($) {
            // Wait for both document and ACF to be ready
            $(document).ready(function() {
                initializeArtistFields();                  
            });

            function loadArtistDetails(termId) {
                if (!termId) {
                    $('[data-name="event_artist_url"] input').val('');
                    return;
                }

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_artist_details',
                        term_id: termId,
                        nonce: '<?php echo esc_js(wp_create_nonce("get_artist_details")); ?>'
                    },
                    success: function(response) {
                        var $artistUrl = $('[data-name="event_artist_url"] input');
                        if (response.success && response.data && response.data.url) {
                            $artistUrl.val(response.data.url);
                        } else {
                            $artistUrl.val('');
                        }
                    },
                    error: function() {
                        $('[data-name="event_artist_url"] input').val('');
                    }
                });
            }

            function initializeArtistFields() {               
                var $artistSelect = $('[data-name="event_artist"] select');
                var initialArtist = $artistSelect.val();
                
                // Load initial artist details
                if (initialArtist) { 
                    loadArtistDetails(initialArtist);
                }

                // Handle artist selection change
                $artistSelect.on('change', function() {
                    loadArtistDetails($(this).val());
                });
            }
        })(jQuery);
        </script>
        <?php
    }
    
    /**
     * Update artist field choices
     */
    public static function update_artist_field_choices($field) {
        $field['choices'] = array();
        
        $artists = get_terms(array(
            'taxonomy' => 'artist',
            'hide_empty' => false,
        ));
        
        if (!empty($artists) && !is_wp_error($artists)) {
            foreach ($artists as $artist) {
                $field['choices'][$artist->term_id] = esc_html($artist->name);
            }
        }
        
        return $field;
    }
}

// Initialize artist handler
add_action('init', array('Vizou_Events_Artist_Handler', 'register'));