<?php

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Import_Export {
    public static function register() {
        add_action('admin_menu', [__CLASS__, 'add_tools_page']);
        add_action('admin_post_vizou_export_events', [__CLASS__, 'handle_export']);
        add_action('admin_post_vizou_import_events', [__CLASS__, 'handle_import']);
    }

    public static function add_tools_page() {
        add_submenu_page(
            'edit.php?post_type=event',
            'Events Tools',
            'Tools',
            'manage_options',
            'vizou-events-tools',
            [__CLASS__, 'render_tools_page']
        );
    }

    public static function render_tools_page() {
        if (isset($_GET['imported']) && isset($_GET['count'])) {
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'vizou_import_complete')) {
                return;
            }
            
            $imported = sanitize_text_field(wp_unslash($_GET['imported']));
            $count = absint($_GET['count']);
            
            if ($imported === 'success') {
                $message = sprintf(
                    /* translators: %d: number of events that were imported */
                    __('%d events imported successfully.', 'vizou-events-manager'),
                    $count
                );
                echo '<div class="notice notice-success"><p>' . esc_html($message) . '</p></div>';
            }
        }
        ?>
        <div class="wrap">
            <h1>Events Tools</h1>
            
            <div class="card">
                <h2>Import Events</h2>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
                    <?php wp_nonce_field('vizou_import_events', 'vizou_import_nonce'); ?>
                    <input type="hidden" name="action" value="vizou_import_events">
                    <input type="file" name="events_csv" accept=".csv" required>
                    <p class="submit">
                        <input type="submit" class="button button-primary" value="Import Events">
                    </p>
                </form>
            </div>

            <div class="card">
                <h2>Export Events</h2>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('vizou_export_events', 'vizou_export_nonce'); ?>
                    <input type="hidden" name="action" value="vizou_export_events">
                    <p class="submit">
                        <input type="submit" class="button button-primary" value="Export Events">
                    </p>
                </form>
            </div>
        </div>
        <?php
    }

    private static function format_time($time_str) {
        if (empty($time_str)) {
            return '';
        }

        // Strip any 'h' character used in French format
        $time_str = str_replace(['h', 'H'], ':', trim($time_str));
        
        // Handle common formats
        if (preg_match('/^(\d{1,2}):(\d{2})\s*(am|pm|AM|PM)?$/', $time_str, $matches)) {
            $hour = intval($matches[1]);
            $minute = $matches[2];
            $meridiem = $matches[3] ?? '';
            
            // Convert to 24-hour if AM/PM is present
            if (strtolower($meridiem) === 'pm' && $hour < 12) {
                $hour += 12;
            } elseif (strtolower($meridiem) === 'am' && $hour === 12) {
                $hour = 0;
            }
            
            return sprintf('%02d:%02d', $hour, $minute);
        }
        
        // Handle times with dots instead of colons (e.g., "7.30pm")
        if (preg_match('/^(\d{1,2})\.(\d{2})\s*(am|pm|AM|PM)?$/', $time_str, $matches)) {
            $hour = intval($matches[1]);
            $minute = $matches[2];
            $meridiem = $matches[3] ?? '';
            
            if (strtolower($meridiem) === 'pm' && $hour < 12) {
                $hour += 12;
            } elseif (strtolower($meridiem) === 'am' && $hour === 12) {
                $hour = 0;
            }
            
            return sprintf('%02d:%02d', $hour, $minute);
        }
        
        // Try parsing with strtotime as fallback
        $timestamp = strtotime($time_str);
        return $timestamp ? date('H:i', $timestamp) : '';
    }

    public static function handle_import() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized access');
        }

        check_admin_referer('vizou_import_events', 'vizou_import_nonce');

        if (!isset($_FILES['events_csv']) || 
            !isset($_FILES['events_csv']['tmp_name']) || 
            !isset($_FILES['events_csv']['name'])) {
            wp_die('No file uploaded');
        }

        $file_name = sanitize_file_name($_FILES['events_csv']['name']);
        $tmp_name = sanitize_text_field($_FILES['events_csv']['tmp_name']);
        if (!preg_match('/\.csv$/i', $file_name)) {
            wp_die('Invalid file type. Please upload a CSV file.');
        }

        $file = wp_check_filetype_and_ext(
            $tmp_name,
            $file_name
        );
        
        if ($file['type'] !== 'text/csv' && $file['type'] !== 'application/csv') {
            wp_die('Invalid file type. Please upload a CSV file.');
        }

        $csv_content = file_get_contents($tmp_name);
        $csv_data = array_map('str_getcsv', explode("\n", $csv_content));
        $headers = array_shift($csv_data);
        
        $imported_count = 0;
        foreach ($csv_data as $i => $row) {
            if (empty($row) || count($row) !== count($headers)) {
                continue;
            }
            
            $data = array_combine($headers, $row);
            
            // Check for existing event with same date and artist
            $existing_event = get_posts([
                'post_type' => 'event',
                'posts_per_page' => 1,
                'meta_query' => [
                    [
                        'key' => 'event_start',
                        'value' => date('Y-m-d', strtotime($data['Date'])),
                        'compare' => '='
                    ]
                ],
                'tax_query' => [
                    [
                        'taxonomy' => 'artist',
                        'field' => 'name',
                        'terms' => wp_strip_all_tags($data['Artist'])
                    ]
                ]
            ]);

            if (!empty($existing_event)) {
                continue;
            }

            $post_data = array(
                'post_title'   => wp_strip_all_tags($data['Venue']),
                'post_type'    => 'event',
                'post_status'  => 'publish',
                'post_content' => wp_kses_post($data['Notes'] ?? '')
            );

            $post_id = wp_insert_post($post_data);

            if ($post_id) {
                $imported_count++;
                
                update_field('event_start', date('Y-m-d', strtotime($data['Date'])), $post_id);
                
                if (!empty($data['Time'])) {
                    $formatted_time = self::format_time($data['Time']);
                    if ($formatted_time) {
                        update_field('event_time', $formatted_time, $post_id);
                    }
                }

                if (isset($data['End date']) && !empty(trim($data['End date'])) && $data['End date'] !== $data['Date']) {
                    $end_timestamp = strtotime($data['End date']);
                    if ($end_timestamp !== false && $end_timestamp > 0) {
                        update_field('event_duration', ['Multi-day event'], $post_id);
                        update_field('event_end', date('Y-m-d', $end_timestamp), $post_id);
                    }
                }
                
                if (!empty($data['Venue'])) {
                    $venue_name = wp_strip_all_tags($data['Venue']);
                    $venue_term = term_exists($venue_name, 'venue');
                    if (!$venue_term) {
                        $venue_term = wp_insert_term($venue_name, 'venue');
                    }
                    
                    if (!is_wp_error($venue_term)) {
                        wp_set_object_terms($post_id, array((int)$venue_term['term_id']), 'venue');
                        
                        $term_id = $venue_term['term_id'];
                        update_field('event_street', wp_strip_all_tags($data['Address'] ?? ''), 'venue_' . $term_id);
                        update_field('event_city', wp_strip_all_tags($data['City'] ?? ''), 'venue_' . $term_id);
                        update_field('event_province', wp_strip_all_tags($data['State'] ?? ''), 'venue_' . $term_id);
                        update_field('event_country', wp_strip_all_tags($data['Country'] ?? ''), 'venue_' . $term_id);
                        update_field('event_venue_url', esc_url_raw($data['Venue URL'] ?? ''), 'venue_' . $term_id);
                    }
                }

                if (!empty($data['Artist'])) {
                    $artist_name = wp_strip_all_tags($data['Artist']);
                    $artist_term = term_exists($artist_name, 'artist');
                    if (!$artist_term) {
                        $artist_term = wp_insert_term($artist_name, 'artist');
                    }
                    
                    if (!is_wp_error($artist_term)) {
                        wp_set_object_terms($post_id, intval($artist_term['term_id']), 'artist');
                        
                        if (!empty($data['Artist URL'])) {
                            update_term_meta($artist_term['term_id'], 'url', esc_url_raw($data['Artist URL']));
                        }
                    }
                }

                if (!empty($data['Ticket URL'])) {
                    update_field('event_ticket_url', esc_url_raw($data['Ticket URL']), $post_id);
                }

                if (!empty($data['Status'])) {
                    $status_map = [
                        'active' => 'Active',
                        'Active' => 'Active',
                        'cancelled' => 'Cancelled',
                        'Cancelled' => 'Cancelled',
                        'sold out' => 'Sold out',
                        'Sold Out' => 'Sold out',
                        'reschedule' => 'To be rescheduled',
                        'Reschedule' => 'To be rescheduled'
                    ];
                    $status = isset($status_map[trim($data['Status'])]) ? $status_map[trim($data['Status'])] : 'Active';
                    update_field('event_status', $status, $post_id);
                }

                if (!empty($data['Tour'])) {
                    $tour_name = wp_strip_all_tags($data['Tour']);
                    $tour_term = term_exists($tour_name, 'tour');
                    if (!$tour_term) {
                        $tour_term = wp_insert_term($tour_name, 'tour');
                    }
                    
                    if (!is_wp_error($tour_term)) {
                        wp_set_object_terms($post_id, intval($tour_term['term_id']), 'tour');
                    }
                }
            }
        }
        
        wp_redirect(add_query_arg(
            [
                'imported' => 'success',
                'count' => $imported_count,
                '_wpnonce' => wp_create_nonce('vizou_import_complete')
            ],
            wp_get_referer()
        ));
        exit;
    }

    public static function handle_export() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized access');
        }

        check_admin_referer('vizou_export_events', 'vizou_export_nonce');

        $events = get_posts([
            'post_type' => 'event',
            'posts_per_page' => -1,
            'orderby' => 'meta_value',
            'meta_key' => 'event_start',
            'order' => 'ASC'
        ]);

        $csv_rows = [];
        $headers = ['Date', 'Time', 'End date', 'Artist', 'Artist URL', 'Tour', 'Venue', 'Address', 'City', 'State', 'Country', 'Venue URL', 'Status', 'Ticket URL', 'Notes'];
        $csv_rows[] = $headers;

        foreach ($events as $event) {
            $venue_terms = wp_get_post_terms($event->ID, 'venue');
            $artist_terms = wp_get_post_terms($event->ID, 'artist');
            $tour_terms = wp_get_post_terms($event->ID, 'tour');
            
            $venue_id = !empty($venue_terms) ? 'venue_' . $venue_terms[0]->term_id : '';
            
            $artist_name = !empty($artist_terms) ? html_entity_decode($artist_terms[0]->name, ENT_QUOTES | ENT_HTML5, 'UTF-8') : '';
            $venue_name = !empty($venue_terms) ? html_entity_decode($venue_terms[0]->name, ENT_QUOTES | ENT_HTML5, 'UTF-8') : '';
            $tour_name = !empty($tour_terms) ? html_entity_decode($tour_terms[0]->name, ENT_QUOTES | ENT_HTML5, 'UTF-8') : '';
            
            $row = [
                get_field('event_start', $event->ID),
                get_field('event_time', $event->ID),
                get_field('event_end', $event->ID),
                $artist_name,
                !empty($artist_terms) ? get_term_meta($artist_terms[0]->term_id, 'url', true) : '',
                $tour_name,
                $venue_name,
                html_entity_decode(get_field('event_street', $venue_id), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                html_entity_decode(get_field('event_city', $venue_id), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                html_entity_decode(get_field('event_province', $venue_id), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                html_entity_decode(get_field('event_country', $venue_id), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                get_field('event_venue_url', $venue_id),
                get_field('event_status', $event->ID),
                get_field('event_ticket_url', $event->ID),
                html_entity_decode($event->post_content, ENT_QUOTES | ENT_HTML5, 'UTF-8')
            ];

            $csv_rows[] = $row;
        }

        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="vizou-events-export-' . current_time('Y-m-d') . '.csv"');
        
        // Add UTF-8 BOM for Excel compatibility
        echo "\xEF\xBB\xBF";

        $fp = fopen('php://output', 'w');
        foreach ($csv_rows as $row) {
            fputcsv($fp, $row);
        }
        fclose($fp);
        exit;
    }
}