<?php
/**
 * Handle custom permalink structure for events
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Permalink_Handler {
    private static $instance = null;
    private static $is_filtering = false;

    public static function register() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_filter('post_type_link', [$this, 'filter_event_permalink'], 10, 2);
        add_filter('template_include', [$this, 'template_include']);
        
        // Add action to flush rewrite rules when needed
        add_action('after_switch_theme', [$this, 'flush_rewrite_rules']);
        add_action('activated_plugin', [$this, 'flush_rewrite_rules']);
        add_action('deactivated_plugin', [$this, 'flush_rewrite_rules']);
    }

    public function flush_rewrite_rules() {
        flush_rewrite_rules();
    }

    public function filter_event_permalink($post_link, $post) {
        if (self::$is_filtering || !is_object($post) || $post->post_type !== 'event') {
            return $post_link;
        }

        self::$is_filtering = true;

        // Get event start date - ACF returns in d/m/Y format
        $start_date = get_field('event_start', $post->ID);
        if (!$start_date) {
            self::$is_filtering = false;
            return $post_link;
        }

        // Convert d/m/Y to Y-m-d for better compatibility with default WordPress date functions
        $date_parts = explode('/', $start_date);
        if (count($date_parts) !== 3) {
            self::$is_filtering = false;
            return $post_link;
        }

        // Create a DateTime object to work with WordPress's default permalink structure
        $date_obj = new DateTime();
        $date_obj->setDate(intval($date_parts[2]), intval($date_parts[1]), intval($date_parts[0]));
        
        // For "Month and name" permalink structure, we should respect that structure
        // This will fit into the /YYYY/MM/post-name/ format that WordPress expects
        $year = $date_obj->format('Y');
        $month = $date_obj->format('m');
        
        // Leave the basic permalink structure intact, but add a query parameter to identify event date
        $permalink = $post_link . '?event_date=' . urlencode($start_date);
        
        self::$is_filtering = false;
        return $permalink;
    }

    public function template_include($template) {
        // Get the global post object
        global $post;
        
        // Check if we're viewing a single event
        if (!is_singular('event')) {
            return $template;
        }
        
        // Check template hierarchy in order of preference
        $templates = array(
            'single-event.php',                      // Theme root
            'vizou-events/vizou-single-event.php',   // Theme custom directory
        );
        
        // Look for a template in the theme
        $theme_template = locate_template($templates);
        if ($theme_template) {
            return $theme_template;
        }
        
        // Fallback to plugin template
        $plugin_template = plugin_dir_path(dirname(__FILE__)) . 'templates/vizou-single-event.php';
        if (file_exists($plugin_template)) {
            return $plugin_template;
        }
        
        // Return original template if none of our options are available
        return $template;
    }
}

// Initialize permalink handler
add_action('init', ['Vizou_Events_Permalink_Handler', 'register']);