<?php
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Settings {
    private static $options_name = 'vizou_events_options';
    private static $default_statuses = [
        'active' => 'Active',
        'cancelled' => 'Cancelled',
        'sold_out' => 'Sold out',
        'reschedule' => 'To be confirmed'
    ];

    private static $default_labels = [
        'tickets' => 'Tickets'
    ];

    public static function register() {
        add_action('admin_menu', [__CLASS__, 'add_settings_page']);
        add_action('admin_init', [__CLASS__, 'register_settings']);
    }

    public static function add_settings_page() {
        add_submenu_page(
            'edit.php?post_type=event',
            esc_html__('Events Settings', 'vizou-events-manager'),
            esc_html__('Settings', 'vizou-events-manager'),
            'manage_options',  // WordPress will handle the capability check
            'vizou-events-settings',
            [__CLASS__, 'render_settings_page']
        );
    }

// Register settings options and sections

    public static function register_settings() {
        // Move capability check here
        if (!current_user_can('manage_options')) {
            return;
        }

        // Define default options explicitly
        $defaults = [
            'no_events_text' => esc_html__('More events to come!', 'vizou-events-manager'),
            'label_tickets' => self::$default_labels['tickets'],
            'request_show_text' => esc_html__('Request a show near you', 'vizou-events-manager'),
            'request_show_url' => '',
            'disable_default_css' => 0,
            'delete_on_uninstall' => 0,
            'display_format' => 'venue_first'
        ];

        // Add default status labels
        foreach (self::$default_statuses as $status_key => $default) {
            $defaults['status_' . sanitize_key($status_key)] = sanitize_text_field($default);
        }

        register_setting(
            self::$options_name, 
            self::$options_name,
            [
                'sanitize_callback' => [__CLASS__, 'sanitize_options'],
                'default' => $defaults,
                'type' => 'array'
            ]
        );

        add_settings_section(
            'vizou_events_main',
            esc_html__('GENERAL SETTINGS', 'vizou-events-manager'),
            [__CLASS__, 'render_general_section_description'],
            'vizou-events-settings'
        );

        add_settings_section(
            'vizou_events_status',
            esc_html__('STATUS LABELS', 'vizou-events-manager'),
            [__CLASS__, 'render_status_section_description'],
            'vizou-events-settings'
        );

        add_settings_section(
            'vizou_events_display',
            esc_html__('DISPLAY OPTIONS', 'vizou-events-manager'),
            null,
            'vizou-events-settings'
        );

        add_settings_section(
            'vizou_events_danger',
            esc_html__('DANGER ZONE', 'vizou-events-manager'),
            null,
            'vizou-events-settings'
        );

// Register settings fields

        foreach (self::$default_statuses as $status_key => $default) {
            add_settings_field(
                'status_' . sanitize_key($status_key),
                esc_html($default),
                [__CLASS__, 'render_status_field'],
                'vizou-events-settings',
                'vizou_events_status',
                [
                    'status_key' => sanitize_key($status_key),
                    'default' => sanitize_text_field($default)
                ]
            );
        }

        add_settings_field(
            'no_events_text',
            esc_html__('No events text', 'vizou-events-manager'),
            [__CLASS__, 'render_no_events_field'],
            'vizou-events-settings',
            'vizou_events_main'
        );

        add_settings_field(
            'label_tickets',
            esc_html__('Ticket button text', 'vizou-events-manager'),
            [__CLASS__, 'render_label_field'],
            'vizou-events-settings',
            'vizou_events_main',
            [
                'field_id' => sanitize_key('tickets'), 
                'default' => sanitize_text_field(self::$default_labels['tickets'])
            ]
        );

        add_settings_field(
            'request_show_text',
            esc_html__('Request show text', 'vizou-events-manager'),
            [__CLASS__, 'render_request_show_text_field'],
            'vizou-events-settings',
            'vizou_events_main'
        );

        add_settings_field(
            'request_show_url',
            esc_html__('Request show URL', 'vizou-events-manager'),
            [__CLASS__, 'render_request_show_url_field'],
            'vizou-events-settings',
            'vizou_events_main'
        );

        add_settings_field(
            'display_format',
            esc_html__('Display format', 'vizou-events-manager'),
            [__CLASS__, 'render_display_format_field'],
            'vizou-events-settings',
            'vizou_events_display'
        );

        add_settings_field(
            'disable_default_css',
            esc_html__('CSS', 'vizou-events-manager'),
            [__CLASS__, 'render_css_checkbox'],
            'vizou-events-settings',
            'vizou_events_display'
        );

        add_settings_field(
            'delete_on_uninstall',
            esc_html__('Clean uninstall', 'vizou-events-manager'),
            [__CLASS__, 'render_delete_checkbox'],
            'vizou-events-settings',
            'vizou_events_danger'
        );
    }

// Sanitize settings options

    public static function sanitize_options($input) {
        if (!current_user_can('manage_options') || !is_array($input)) {
            return false;
        }
        
        $sanitized = [];
        
        // Sanitize text fields with stricter validation
        if (isset($input['no_events_text'])) {
            $allowed_html = array(
                'em' => array(),
                'strong' => array(),
                'span' => array('class' => array())
            );
            $sanitized['no_events_text'] = wp_kses($input['no_events_text'], $allowed_html);
        }
        
        if (isset($input['label_tickets'])) {
            $sanitized['label_tickets'] = sanitize_text_field($input['label_tickets']);
        }

        if (isset($input['request_show_text'])) {
            $sanitized['request_show_text'] = sanitize_text_field($input['request_show_text']);
        }

        if (isset($input['request_show_url'])) {
            $url = trim($input['request_show_url']);
            $sanitized['request_show_url'] = !empty($url) ? esc_url_raw($url) : '';
        }
        
        // Sanitize status labels with validation
        foreach (self::$default_statuses as $status_key => $_) {
            $key = 'status_' . sanitize_key($status_key);
            if (isset($input[$key])) {
                $sanitized[$key] = sanitize_text_field($input[$key]);
            } else {
                $sanitized[$key] = self::$default_statuses[$status_key];
            }
        }
        
        // Sanitize checkboxes with explicit boolean casting
        $sanitized['disable_default_css'] = !empty($input['disable_default_css']) ? 1 : 0;
        $sanitized['delete_on_uninstall'] = !empty($input['delete_on_uninstall']) ? 1 : 0;

         $sanitized['display_format'] = in_array($input['display_format'], ['venue_first', 'artist_first'], true) ? 
            $input['display_format'] : 'venue_first';
        
        return $sanitized;
    }

// Render settings options

    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'vizou-events-manager'));
        }
        ?>
        <div class="wrap">
            <?php settings_errors(); ?>
            <h1><?php echo esc_html__('Events Settings', 'vizou-events-manager'); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields(self::$options_name);
                do_settings_sections('vizou-events-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public static function render_status_section_description() {
        echo '<p>' . esc_html__('Customize how event status appears on the frontend.', 'vizou-events-manager') . '</p>';
    }

    public static function render_general_section_description() {
        echo '<p>' . esc_html__('Customize the "no events" message, button text, and request show link.', 'vizou-events-manager') . '</p>';
    }

    public static function render_status_field($args) {
        $options = get_option(self::$options_name);
        $status_key = sanitize_key($args['status_key']);
        $default = sanitize_text_field($args['default']);
        
        $value = isset($options['status_' . $status_key]) ? $options['status_' . $status_key] : $default;
        ?>
        <input type="text" 
               name="<?php echo esc_attr(self::$options_name); ?>[status_<?php echo esc_attr($status_key); ?>]"
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text">
        <?php
    }

    public static function render_no_events_field() {
        $options = get_option(self::$options_name);
        $value = isset($options['no_events_text']) ? $options['no_events_text'] : esc_html__('More to come!', 'vizou-events-manager');
        ?>
        <input type="text" 
               name="<?php echo esc_attr(self::$options_name); ?>[no_events_text]" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text">
        <?php
    }

    public static function render_label_field($args) {
        $options = get_option(self::$options_name);
        $field_id = sanitize_key($args['field_id']);
        $default = sanitize_text_field($args['default']);
        
        $value = isset($options['label_' . $field_id]) ? $options['label_' . $field_id] : $default;
        ?>
        <input type="text" 
               name="<?php echo esc_attr(self::$options_name); ?>[label_<?php echo esc_attr($field_id); ?>]" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text">
        <?php
    }

    public static function render_request_show_text_field() {
        $options = get_option(self::$options_name);
        $value = isset($options['request_show_text']) ? $options['request_show_text'] : esc_html__('Request a show near you', 'vizou-events-manager');
        ?>
        <input type="text" 
               name="<?php echo esc_attr(self::$options_name); ?>[request_show_text]" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text">
        <p class="description"><?php esc_html_e('This text only appears when a URL is provided.', 'vizou-events-manager'); ?></p>
        <?php
    }

    public static function render_request_show_url_field() {
        $options = get_option(self::$options_name);
        $value = isset($options['request_show_url']) ? $options['request_show_url'] : '';
        ?>
        <input type="url" 
               name="<?php echo esc_attr(self::$options_name); ?>[request_show_url]" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text"
               placeholder="https://">
        <p class="description"><?php esc_html_e('The request show text will only appear if this URL is provided.', 'vizou-events-manager'); ?></p>
        <?php
    }

        public static function render_display_format_field() {
        $options = get_option(self::$options_name);
        $current = isset($options['display_format']) ? $options['display_format'] : 'venue_first';
        ?>
        <fieldset>
            <legend class="screen-reader-text"><?php esc_html_e('Event display format', 'vizou-events-manager'); ?></legend>
            <label>
                <input type="radio" name="<?php echo esc_attr(self::$options_name); ?>[display_format]" 
                    value="venue_first" <?php checked('venue_first', $current); ?> />
                <?php esc_html_e('Venue first (default) - Show venue before artist', 'vizou-events-manager'); ?>
            </label>
            <br />
            <label>
                <input type="radio" name="<?php echo esc_attr(self::$options_name); ?>[display_format]" 
                    value="artist_first" <?php checked('artist_first', $current); ?> />
                <?php esc_html_e('Artist first - Show artist before venue', 'vizou-events-manager'); ?>
            </label>           
        </fieldset>
        <?php
    }

    public static function render_css_checkbox() {
        $options = get_option(self::$options_name);
        $checked = !empty($options['disable_default_css']);
        ?>
        <input type="checkbox" 
               name="<?php echo esc_attr(self::$options_name); ?>[disable_default_css]" 
               value="1" 
               <?php checked(1, $checked); ?>>
        <span class="description">
            <?php esc_html_e('Disable default CSS (use theme style.css instead)', 'vizou-events-manager'); ?>
        </span>
        <?php
    }

    public static function render_delete_checkbox() {
        $options = get_option(self::$options_name);
        $checked = !empty($options['delete_on_uninstall']);
        ?>
        <input type="checkbox" 
               name="<?php echo esc_attr(self::$options_name); ?>[delete_on_uninstall]" 
               value="1" 
               <?php checked(1, $checked); ?>>
        <span class="description">
            <?php esc_html_e('Delete all plugin data when uninstalling', 'vizou-events-manager'); ?>
        </span>
        <?php
    }

    public static function get_status_label($status) {
        if (empty($status)) {
            return '';
        }
        
        $options = get_option(self::$options_name);
        $status_key = 'status_' . sanitize_key(strtolower($status));
        
        if (isset($options[$status_key])) {
            return esc_html($options[$status_key]);
        }
        
        return isset(self::$default_statuses[strtolower($status)]) 
            ? esc_html(self::$default_statuses[strtolower($status)]) 
            : esc_html($status);
    }

    public static function is_cancelled($status) {
        if (empty($status)) {
            return false;
        }
        return strtolower(trim($status)) === 'cancelled';
    }

    public static function get_option($key) {
        if (empty($key)) {
            return null;
        }
        
        $options = get_option(self::$options_name);
        if (!is_array($options)) {
            return null;
        }
        
        return isset($options[$key]) ? sanitize_text_field($options[$key]) : null;
    }
}