<?php
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Shortcodes {
    public static function register() {
        // Register the old shortcode for backward compatibility - to be removed soon!
        add_shortcode('showdates', array(__CLASS__, 'show_dates_shortcode'));    
        // Register the new, more intuitive shortcode name
        add_shortcode('vizou-events', array(__CLASS__, 'show_dates_shortcode'));
    }
    
    public static function show_dates_shortcode($atts, $content = null) {
        wp_enqueue_style('vizou-events-manager');
        
        $today = current_time('Y-m-d');
        $atts = shortcode_atts(array(
            'show' => 'upcoming',
            'year' => '',
            'order' => 'asc',
            'limit' => -1,
            'tour' => '',
            'exclude_tour' => '',
            'artist' => '',
            'requestshow' => ''
        ), $atts, 'showdates');

        // Check if requestshow is the only meaningful parameter
        $is_requestshow_only = !empty($atts['requestshow']) && 
                              $atts['show'] === 'upcoming' && 
                              empty($atts['year']) && 
                              $atts['order'] === 'asc' && 
                              $atts['limit'] == -1 && 
                              empty($atts['tour']) && 
                              empty($atts['exclude_tour']) && 
                              empty($atts['artist']);

        // If requestshow is the only parameter, show only the request link
        if ($is_requestshow_only && function_exists('vizou_request_show_link')) {
            $request_link = vizou_request_show_link(false);
            if (!empty($request_link)) {
                return '<div class="vizou-events-container"><div class="vizou-request-show-wrapper">' . wp_kses_post($request_link) . '</div></div>';
            }
            return ''; // Return empty if no URL configured
        }

        $args = array(
            'post_type' => 'event',
            'posts_per_page' => intval($atts['limit']),
            'meta_key' => 'event_start',
            'orderby' => 'meta_value',
            'order' => (strtolower($atts['order']) === 'desc') ? 'DESC' : 'ASC',
            'meta_query' => array(
                'relation' => 'AND'
            ),
            'tax_query' => array(
                'relation' => 'AND'
            )
        );

        // Handle multiple tours (include)
        if (!empty($atts['tour'])) {
            $tour_ids = self::parse_comma_separated_ids($atts['tour']);
            if (!empty($tour_ids)) {
                $args['tax_query'][] = array(
                    'taxonomy' => 'tour',
                    'field' => 'term_id',
                    'terms' => $tour_ids
                );
            }
        }

        // Handle multiple tours (exclude)
        if (!empty($atts['exclude_tour'])) {
            $exclude_tour_ids = self::parse_comma_separated_ids($atts['exclude_tour']);
            if (!empty($exclude_tour_ids)) {
                $args['tax_query'][] = array(
                    'taxonomy' => 'tour',
                    'field' => 'term_id',
                    'terms' => $exclude_tour_ids,
                    'operator' => 'NOT IN'
                );
            }
        }

        // Handle multiple artists
        if (!empty($atts['artist'])) {
            $artist_ids = self::parse_comma_separated_ids($atts['artist']);
            if (!empty($artist_ids)) {
                $args['tax_query'][] = array(
                    'taxonomy' => 'artist',
                    'field' => 'term_id',
                    'terms' => $artist_ids
                );
            }
        }

        // Handle date filtering based on 'show' parameter
        switch($atts['show']) {
            case 'past':
                $args['meta_query'][] = array(
                    'key' => 'event_start',
                    'compare' => '<',
                    'value' => $today,
                    'type' => 'DATE'
                );
                $args['order'] = 'DESC';
                break;
            case 'all':
                break;
            case 'upcoming':
            default:
                $args['meta_query'][] = array(
                    'key' => 'event_start',
                    'compare' => '>=',
                    'value' => $today,
                    'type' => 'DATE'
                );
                break;
        }

        // Handle multiple years
        if (!empty($atts['year'])) {
            $years = self::parse_comma_separated_years($atts['year']);
            if (!empty($years)) {
                if (count($years) === 1) {
                    // Single year - use existing logic
                    $year = $years[0];
                    $args['meta_query'][] = array(
                        'key' => 'event_start',
                        'compare' => 'BETWEEN',
                        'value' => array($year . '-01-01', $year . '-12-31'),
                        'type' => 'DATE'
                    );
                } else {
                    // Multiple years - create OR relationship
                    $year_queries = array('relation' => 'OR');
                    foreach ($years as $year) {
                        $year_queries[] = array(
                            'key' => 'event_start',
                            'compare' => 'BETWEEN',
                            'value' => array($year . '-01-01', $year . '-12-31'),
                            'type' => 'DATE'
                        );
                    }
                    $args['meta_query'][] = $year_queries;
                }
            }
        }

        $query = new WP_Query($args);
        
        ob_start();
        echo '<div class="vizou-events-container">';

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                
                // First check for theme override template
                $theme_template = locate_template('vizou-events/vizou-events-list.php');
                
                if ($theme_template && file_exists($theme_template)) {
                    // Use the theme's override template
                    include $theme_template;
                } else {
                    // If no theme override exists, use the plugin's default template
                    $default_template = vizou_events_manager()->get_template_path('vizou-events-list.php');
                    if ($default_template && file_exists($default_template)) {
                        include $default_template;
                    } else {
                        // Emergency fallback if default template is not found
                        echo "<!-- Template not found for event ID: " . esc_html(get_the_ID()) . " -->";
                    }
                }
            }
        } else {
            echo '<p class="no-events">' . esc_html(Vizou_Events_Settings::get_option('no_events_text')) . '</p>';
        }

        // Add request show link if explicitly requested and available
        if (!empty($atts['requestshow']) && function_exists('vizou_request_show_link')) {
            $request_link = vizou_request_show_link(false); // Get without echoing
            if (!empty($request_link)) {
                echo '<div class="vizou-request-show-wrapper">' . wp_kses_post($request_link) . '</div>';
            }
        }

        // Close the container
        echo '</div>';

        wp_reset_postdata();
        return ob_get_clean();
    }

    /**
     * Parse comma-separated IDs and return array of sanitized integers
     *
     * @param string $input Comma-separated string of IDs
     * @return array Array of positive integers
     */
    private static function parse_comma_separated_ids($input) {
        if (empty($input)) {
            return array();
        }

        // Split by comma and clean up
        $ids = explode(',', $input);
        $clean_ids = array();

        foreach ($ids as $id) {
            $id = trim($id);
            if (is_numeric($id) && $id > 0) {
                $clean_ids[] = intval($id);
            }
        }

        return array_unique($clean_ids);
    }

    /**
     * Parse comma-separated years and return array of valid 4-digit years
     *
     * @param string $input Comma-separated string of years
     * @return array Array of valid 4-digit years
     */
    private static function parse_comma_separated_years($input) {
        if (empty($input)) {
            return array();
        }

        // Split by comma and clean up
        $years = explode(',', $input);
        $clean_years = array();

        foreach ($years as $year) {
            $year = trim($year);
            // Validate as 4-digit year (reasonable range: 1900-2100)
            if (preg_match('/^\d{4}$/', $year) && $year >= 1900 && $year <= 2100) {
                $clean_years[] = $year;
            }
        }

        return array_unique($clean_years);
    }
}