<?php
/**
 * Register custom taxonomies
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Taxonomies {
    public static function register() {
        self::register_venue_taxonomy();
        self::register_artist_taxonomy();
        self::register_tour_taxonomy();

        // Add term ID and field columns
        add_filter('manage_edit-venue_columns', [__CLASS__, 'add_term_columns']);
        add_filter('manage_venue_custom_column', [__CLASS__, 'populate_term_columns'], 10, 3);
        add_filter('manage_edit-artist_columns', [__CLASS__, 'add_term_columns']);
        add_filter('manage_artist_custom_column', [__CLASS__, 'populate_term_columns'], 10, 3);
        add_filter('manage_edit-tour_columns', [__CLASS__, 'add_term_id_column']);
        add_filter('manage_tour_custom_column', [__CLASS__, 'populate_term_id_column'], 10, 3);
        
        // Hide duplicate venues column in All Events view
        add_filter('manage_event_posts_columns', [__CLASS__, 'remove_venue_column']);

        // Add fields to taxonomy forms
        add_action('venue_add_form_fields', [__CLASS__, 'add_venue_fields']);
        add_action('venue_edit_form_fields', [__CLASS__, 'edit_venue_fields']);
        add_action('artist_add_form_fields', [__CLASS__, 'add_url_field']);
        add_action('artist_edit_form_fields', [__CLASS__, 'edit_url_field']);

        // Save fields
        add_action('created_venue', [__CLASS__, 'save_venue_fields']);
        add_action('edited_venue', [__CLASS__, 'save_venue_fields']);
        add_action('created_artist', [__CLASS__, 'save_url_field']);
        add_action('edited_artist', [__CLASS__, 'save_url_field']);

        // Sync ACF venue fields with taxonomy meta
        add_filter('acf/update_value/name=event_street', [__CLASS__, 'sync_event_street'], 10, 3);
        add_filter('acf/update_value/name=event_city', [__CLASS__, 'sync_event_city'], 10, 3);
        add_filter('acf/update_value/name=event_province', [__CLASS__, 'sync_event_province'], 10, 3);
        add_filter('acf/update_value/name=event_country', [__CLASS__, 'sync_event_country'], 10, 3);
        add_filter('acf/update_value/name=event_venue_url', [__CLASS__, 'sync_venue_url'], 10, 3);
        add_filter('acf/load_value/name=event_street', [__CLASS__, 'load_event_street'], 10, 3);
        add_filter('acf/load_value/name=event_city', [__CLASS__, 'load_event_city'], 10, 3);
        add_filter('acf/load_value/name=event_province', [__CLASS__, 'load_event_province'], 10, 3);
        add_filter('acf/load_value/name=event_country', [__CLASS__, 'load_event_country'], 10, 3);
        add_filter('acf/load_value/name=event_venue_url', [__CLASS__, 'load_venue_url'], 10, 3);
        
        // Sync ACF artist fields with taxonomy meta
        add_filter('acf/update_value/name=event_artist_url', [__CLASS__, 'sync_artist_url'], 10, 3);     
        add_filter('acf/load_value/name=event_artist_url', [__CLASS__, 'load_artist_url'], 10, 3);       
    }

    /* VENUE FIELDS */

    // ADD venue fields  to taxonomy
    public static function add_venue_fields($taxonomy) {
        if ($taxonomy !== 'venue') {
            return;
        }
    
        wp_nonce_field('vizou_events_term_meta', 'vizou_events_term_meta_nonce');
        ?>
        <div class="form-field">
            <label for="term_url">Website URL</label>
            <input type="url" name="term_url" id="term_url" value="" placeholder="https://" />
            <p>The URL for this venue.</p>
        </div>
        <div class="form-field">
            <label for="term_street">Street Address</label>
            <input type="text" name="term_street" id="term_street" value="" />
            <p>The street address of this venue.</p>
        </div>
        <div class="form-field">
            <label for="term_city">City</label>
            <input type="text" name="term_city" id="term_city" value="" />
            <p>The city where this venue is located.</p>
        </div>
        <div class="form-field">
            <label for="term_province">Province/State</label>
            <input type="text" name="term_province" id="term_province" value="" />
            <p>The province or state of this venue.</p>
        </div>
        <div class="form-field">
            <label for="term_country">Country</label>
            <input type="text" name="term_country" id="term_country" value="" />
            <p>The country where this venue is located.</p>
        </div>
        <?php
    }

    // EDIT venue fields
    public static function edit_venue_fields($term) {
        if ($term->taxonomy !== 'venue') {
            return;
        }
        
        wp_nonce_field('vizou_events_term_meta', 'vizou_events_term_meta_nonce');
        $url = get_term_meta($term->term_id, 'url', true);
        // Try to get street with event_ prefix first, fall back to without prefix
        $street = get_term_meta($term->term_id, 'event_street', true);
        if (empty($street)) {
            $street = get_term_meta($term->term_id, 'street', true);
        }
        // Try to get city with event_ prefix first, fall back to without prefix
        $city = get_term_meta($term->term_id, 'event_city', true);
        if (empty($city)) {
            $city = get_term_meta($term->term_id, 'city', true);
        }
        // Try to get province with event_ prefix first, fall back to without prefix
        $province = get_term_meta($term->term_id, 'event_province', true);
        if (empty($province)) {
            $province = get_term_meta($term->term_id, 'province', true);
        }
        // Try to get country with event_ prefix first, fall back to without prefix
        $country = get_term_meta($term->term_id, 'event_country', true);
        if (empty($country)) {
            $country = get_term_meta($term->term_id, 'country', true);
        }
        ?>
        <tr class="form-field">
            <th><label for="term_url">Website URL</label></th>
            <td>
                <input type="url" name="term_url" id="term_url" value="<?php echo esc_attr($url); ?>" placeholder="https://" />
                <p class="description">The URL for this venue.</p>
            </td>
        </tr>
        <tr class="form-field">
            <th><label for="term_street">Street Address</label></th>
            <td>
                <input type="text" name="term_street" id="term_street" value="<?php echo esc_attr($street); ?>" />
                <p class="description">The street address of this venue.</p>
            </td>
        </tr>
        <tr class="form-field">
            <th><label for="term_city">City</label></th>
            <td>
                <input type="text" name="term_city" id="term_city" value="<?php echo esc_attr($city); ?>" />
                <p class="description">The city where this venue is located.</p>
            </td>
        </tr>
        <tr class="form-field">
            <th><label for="term_province">Province/State</label></th>
            <td>
                <input type="text" name="term_province" id="term_province" value="<?php echo esc_attr($province); ?>" />
                <p class="description">The province or state of this venue.</p>
            </td>
        </tr>
        <tr class="form-field">
            <th><label for="term_country">Country</label></th>
            <td>
                <input type="text" name="term_country" id="term_country" value="<?php echo esc_attr($country); ?>" />
                <p class="description">The country where this venue is located.</p>
            </td>
        </tr>
        <?php
    }

    // SAVE venue fields
    public static function save_venue_fields($term_id) {
    // Verify nonce
        if (!isset($_POST['vizou_events_term_meta_nonce'])) {
            return;
        }
        
        $nonce = sanitize_text_field(wp_unslash($_POST['vizou_events_term_meta_nonce']));
        if (!wp_verify_nonce($nonce, 'vizou_events_term_meta')) {
            return;
        }

        // Check if user has permissions
        if (!current_user_can('manage_categories')) {
            return;
        }

        // Save URL
        if (isset($_POST['term_url'])) {
            update_term_meta(
                $term_id,
                'url',
                esc_url_raw(wp_unslash($_POST['term_url']))
            );
        }

        // Save address fields with event_ prefix for standardization
        $fields = [
            'event_street' => 'term_street',
            'event_city' => 'term_city',
            'event_province' => 'term_province',
            'event_country' => 'term_country'
        ];

        foreach ($fields as $meta_key => $field_name) {
            if (isset($_POST[$field_name])) {
                $value = sanitize_text_field(wp_unslash($_POST[$field_name]));
                update_term_meta($term_id, $meta_key, $value);
                
                // Clean up old format if it exists (without event_ prefix)
                $old_key = str_replace('event_', '', $meta_key);
                delete_term_meta($term_id, $old_key);
            }
        }
    }

    // LOAD venue address fields from taxonomy meta to ACF
    public static function load_event_street($value, $post_id, $field) {
        // First try getting venue from ACF field
        $venues = get_field('event_name', $post_id); 
        // If that fails, try getting venue directly from taxonomy terms
        if (empty($venues) || is_wp_error($venues)) {
            $venues = wp_get_post_terms($post_id, 'venue');
        }       
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Try with event_ prefix first (new standardized format)
            $street = get_term_meta($venue_id, 'event_street', true);
            
            // If not found, try without prefix (old format)
            if (empty($street)) {
                $street = get_term_meta($venue_id, 'street', true);
            }
            
            if (!empty($street)) {
                return $street;
            }
        }
        return $value;
    }

    public static function load_event_city($value, $post_id, $field) {
        // Get the venue ID directly
        $venues = get_field('event_name', $post_id);
        
        // If that fails, try getting venue directly from taxonomy terms
        if (empty($venues) || is_wp_error($venues)) {
            $venues = wp_get_post_terms($post_id, 'venue');
        }
        
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Try with event_ prefix first (new standardized format)
            $city = get_term_meta($venue_id, 'event_city', true);
            
            // If not found, try without prefix (old format)
            if (empty($city)) {
                $city = get_term_meta($venue_id, 'city', true);
            }
            
            if (!empty($city)) {
                return $city;
            }
        }
        
        return $value;
    }

    public static function load_event_province($value, $post_id, $field) {
        $venues = get_field('event_name', $post_id);    
        if (empty($venues) || is_wp_error($venues)) {
            $venues = wp_get_post_terms($post_id, 'venue');
        }       
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Try with event_ prefix first (new standardized format)
            $province = get_term_meta($venue_id, 'event_province', true);
            
            // If not found, try without prefix (old format)
            if (empty($province)) {
                $province = get_term_meta($venue_id, 'province', true);
            }
            
            if (!empty($province)) {
                return $province;
            }
        }
        return $value;
    }

    public static function load_event_country($value, $post_id, $field) {
        $venues = get_field('event_name', $post_id);    
        if (empty($venues) || is_wp_error($venues)) {
            $venues = wp_get_post_terms($post_id, 'venue');
        }       
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Try with event_ prefix first (new standardized format)
            $country = get_term_meta($venue_id, 'event_country', true);
            
            // If not found, try without prefix (old format)
            if (empty($country)) {
                $country = get_term_meta($venue_id, 'country', true);
            }
            
            if (!empty($country)) {
                return $country;
            }
        }
        return $value;
    }

    public static function load_venue_url($value, $post_id, $field) {
        $venues = get_field('event_name', $post_id);    
        if (empty($venues) || is_wp_error($venues)) {
            $venues = wp_get_post_terms($post_id, 'venue');
        }       
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Try with event_ prefix first (new standardized format)
            $url = get_term_meta($venue_id, 'event_venue_url', true);
            
            // If not found, try without prefix (old format)
            if (empty($url)) {
                $url = get_term_meta($venue_id, 'url', true);
            }
            
            if (!empty($url)) {
                return $url;
            }
        }
        return $value;
    }

    // SYNC venue fields from ACF to taxonomy meta
    public static function sync_event_street($value, $post_id, $field) {
        if (!current_user_can('edit_post', $post_id)) {
            return $value;
        }

        $venues = get_field('event_name', $post_id);
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Store with event_ prefix as the standard format
            update_term_meta($venue_id, 'event_street', sanitize_text_field(wp_unslash($value)));
            
            // Clean up old format if it exists
            delete_term_meta($venue_id, 'street');
        }
        return $value;
    }

    public static function sync_event_city($value, $post_id, $field) {
        if (!current_user_can('edit_post', $post_id)) {
            return $value;
        }

        $venues = get_field('event_name', $post_id);
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Store with event_ prefix as the standard format
            update_term_meta($venue_id, 'event_city', sanitize_text_field(wp_unslash($value)));
            
            // Clean up old format if it exists
            delete_term_meta($venue_id, 'city');
        }
        return $value;
    }

    public static function sync_event_province($value, $post_id, $field) {
        if (!current_user_can('edit_post', $post_id)) {
            return $value;
        }

        $venues = get_field('event_name', $post_id);
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Store with event_ prefix as the standard format
            update_term_meta($venue_id, 'event_province', sanitize_text_field(wp_unslash($value)));
            
            // Clean up old format if it exists
            delete_term_meta($venue_id, 'province');
        }
        return $value;
    }

    public static function sync_event_country($value, $post_id, $field) {
        if (!current_user_can('edit_post', $post_id)) {
            return $value;
        }

        $venues = get_field('event_name', $post_id);
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Store with event_ prefix as the standard format
            update_term_meta($venue_id, 'event_country', sanitize_text_field(wp_unslash($value)));
            
            // Clean up old format if it exists
            delete_term_meta($venue_id, 'country');
        }
        return $value;
    }

    public static function sync_venue_url($value, $post_id, $field) {
        if (!current_user_can('edit_post', $post_id)) {
            return $value;
        }

        $venues = get_field('event_name', $post_id);
        if (!empty($venues) && !is_wp_error($venues)) {
            $venue_id = is_object($venues) ? $venues->term_id : $venues[0]->term_id;
            
            // Store with event_ prefix as the standard format
            update_term_meta($venue_id, 'event_venue_url', esc_url_raw(wp_unslash($value)));
            
            // Also update the 'url' field for compatibility with other parts of the code
            update_term_meta($venue_id, 'url', esc_url_raw(wp_unslash($value)));
        }
        return $value;
    }

    /* ARTIST FIELDS */

    // ADD artist url field to taxonomy form
    public static function add_url_field($taxonomy) {
        if ($taxonomy !== 'artist') {
            return;
        }
    
        wp_nonce_field('vizou_events_term_url', 'vizou_events_term_url_nonce');
        ?>
        <div class="form-field">
            <label for="term_url">Website URL</label>
            <input type="url" name="term_url" id="term_url" value="" placeholder="https://" />
            <p>The URL for this artist.</p>
        </div>
        <?php
    }

    // EDIT artist url field in taxonomy form
    public static function edit_url_field($term) {
        if ($term->taxonomy !== 'artist') {
            return;
        }
        
        wp_nonce_field('vizou_events_term_url', 'vizou_events_term_url_nonce');
        $url = get_term_meta($term->term_id, 'url', true);
        ?>
        <tr class="form-field">
            <th><label for="term_url">Website URL</label></th>
            <td>
                <input type="url" name="term_url" id="term_url" value="<?php echo esc_attr($url); ?>" placeholder="https://" />
                <p class="description">The URL for this artist.</p>
            </td>
        </tr>
        <?php
    }

    // SAVE artist url field in taxonomy form
    public static function save_url_field($term_id) {
        // Verify nonce
        if (!isset($_POST['vizou_events_term_url_nonce'])) {
            return;
        }
        
        $nonce = sanitize_text_field(wp_unslash($_POST['vizou_events_term_url_nonce']));
        if (!wp_verify_nonce($nonce, 'vizou_events_term_url')) {
            return;
        }

        // Check if user has permissions
        if (!current_user_can('manage_categories')) {
            return;
        }

        if (isset($_POST['term_url'])) {
            update_term_meta(
                $term_id,
                'url',
                esc_url_raw(wp_unslash($_POST['term_url']))
            );
        }
    }

    // LOAD artist url from taxonomy meta to ACF field
    public static function load_artist_url($value, $post_id, $field) {
        $artists = get_field('event_artist', $post_id);
        if (!empty($artists) && !is_wp_error($artists)) {
            $artist_id = is_object($artists) ? $artists->term_id : $artists[0]->term_id;
            $url = get_term_meta($artist_id, 'url', true);
            if (!empty($url)) {
                return $url;
            }
        }
        return $value;
    }

    // SYNC artist url from ACF to taxonomy meta
    public static function sync_artist_url($value, $post_id, $field) {
        if (!current_user_can('edit_post', $post_id)) {
            return $value;
        }

        $artists = get_field('event_artist', $post_id);
        if (!empty($artists) && !is_wp_error($artists)) {
            $artist_id = is_object($artists) ? $artists->term_id : $artists[0]->term_id;
            update_term_meta($artist_id, 'url', esc_url_raw(wp_unslash($value)));
        }
        return $value;
    }

    public static function add_term_columns($columns) {
        $columns['term_id'] = 'ID';
        $columns['url'] = 'Website URL';
        return $columns;
    }

    public static function add_term_id_column($columns) {
        $columns['term_id'] = 'ID';
        return $columns;
    }

    public static function populate_term_columns($content, $column_name, $term_id) {
        if ($column_name === 'term_id') {
            return $term_id;
        }
        if ($column_name === 'url') {
            $url = get_term_meta($term_id, 'url', true);
            if (!empty($url)) {
                return sprintf('<a href="%s" target="_blank">%s</a>', esc_url($url), esc_html($url));
            }
        }
        return $content;
    }

    public static function populate_term_id_column($content, $column_name, $term_id) {
        if ($column_name === 'term_id') {
            return $term_id;
        }
        return $content;
    }

    public static function remove_venue_column($columns) {
        unset($columns['taxonomy-venue']);
        unset($columns['taxonomy-artist']);
        unset($columns['taxonomy-tour']);
        return $columns;
    }
    
    private static function register_venue_taxonomy() {
        $labels = array(
            'name'              => 'Venues',
            'singular_name'     => 'Venue',
            'search_items'      => 'Search Venues',
            'all_items'         => 'Venues',
            'edit_item'         => 'Edit Venue',
            'update_item'       => 'Update Venue',
            'add_new_item'      => 'Add New Venue',
            'new_item_name'     => 'New Venue Name',
            'menu_name'         => 'Venues',
            'not_found'         => 'No Venues Found',
            'back_to_items'     => 'Go to Venues',
            'parent_item'       => null,
            'parent_item_colon' => null
        );

        $args = array(
            'labels'                => $labels,
            'hierarchical'          => false,
            'public'               => true,
            'show_ui'              => true,
            'show_admin_column'    => true,
            'show_in_nav_menus'    => false,
            'show_tagcloud'        => false,
            'show_in_rest'         => true,
            'meta_box_cb'          => false,
            'rewrite'              => array('slug' => 'venue'),
            'capabilities'         => array(
                'manage_terms'  => 'manage_categories',
                'edit_terms'    => 'manage_categories',
                'delete_terms'  => 'manage_categories',
                'assign_terms'  => 'edit_posts'
            ),
            'show_in_quick_edit'   => false
        );

        register_taxonomy('venue', array('event'), $args);
    }
    
    private static function register_artist_taxonomy() {
        $labels = array(
            'name'              => 'Artists',
            'singular_name'     => 'Artist',
            'search_items'      => 'Search Artists',
            'all_items'         => 'Artists',
            'edit_item'         => 'Edit Artist',
            'update_item'       => 'Update Artist',
            'add_new_item'      => 'Add New Artist',
            'new_item_name'     => 'New Artist Name',
            'menu_name'         => 'Artists',
            'not_found'         => 'No Artists Found',
            'back_to_items'     => 'Go to Artists',
            'parent_item'       => null,
            'parent_item_colon' => null
        );

        $args = array(
            'labels'                => $labels,
            'hierarchical'          => false,
            'public'               => true,
            'show_ui'              => true,
            'show_admin_column'    => true,
            'show_in_nav_menus'    => false,
            'show_tagcloud'        => false,
            'show_in_rest'         => true,
            'meta_box_cb'          => false,
            'rewrite'              => array('slug' => 'artist'),
            'capabilities'         => array(
                'manage_terms'  => 'manage_categories',
                'edit_terms'    => 'manage_categories',
                'delete_terms'  => 'manage_categories',
                'assign_terms'  => 'edit_posts'
            ),
            'show_in_quick_edit'   => false
        );

        register_taxonomy('artist', array('event'), $args);
    }
    
    private static function register_tour_taxonomy() {
        $labels = array(
            'name'              => 'Tours',
            'singular_name'     => 'Tour',
            'search_items'      => 'Search Tours',
            'all_items'         => 'Tours',
            'edit_item'         => 'Edit Tour',
            'update_item'       => 'Update Tour',
            'add_new_item'      => 'Add New Tour',
            'new_item_name'     => 'New Tour Name',
            'menu_name'         => 'Tours',
            'not_found'         => 'No Tours Found',
            'back_to_items'     => 'Go to Tours',
            'parent_item'       => null,
            'parent_item_colon' => null
        );

        $args = array(
            'labels'                => $labels,
            'hierarchical'          => false,
            'public'               => true,
            'show_ui'              => true,
            'show_admin_column'    => true,
            'show_in_nav_menus'    => false,
            'show_tagcloud'        => false,
            'show_in_rest'         => true,
            'meta_box_cb'          => false,
            'rewrite'              => array('slug' => 'tour'),
            'capabilities'         => array(
                'manage_terms'  => 'manage_categories',
                'edit_terms'    => 'manage_categories',
                'delete_terms'  => 'manage_categories',
                'assign_terms'  => 'edit_posts'
            ),
            'show_in_quick_edit'   => false
        );

        register_taxonomy('tour', array('event'), $args);
    }
}