<?php
/**
 * Vizou Events Template Tags
 *
 * These functions can be used in theme templates to customize the display of events.
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Display or get the event date
 *
 * @param int $post_id Optional post ID (defaults to current post)
 * @param bool $echo Whether to display or return
 * @return string The formatted date
 */
function vizou_event_date($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $date = get_field('event_start', $post_id);
    
    if (!$date) return '';
    
    // Convert from stored d/m/Y format to a timestamp
    $date_obj = DateTime::createFromFormat('d/m/Y', $date);
    if (!$date_obj) return '';
    
    // Format using WordPress date format setting
    $formatted = $date_obj->format(get_option('date_format'));
    
    if ($echo) {
        echo esc_html($formatted);
    }
    return $formatted;
}

/**
 * Display or get the event end date
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The formatted end date or empty if same as start date
 */
function vizou_event_end_date($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $start_date = get_field('event_start', $post_id);
    $end_date = get_field('event_end', $post_id);
    
    // If no end date or same as start date, return empty
    if (!$end_date || $end_date == $start_date) return '';
    
    // Convert from stored d/m/Y format to a timestamp
    $date_obj = DateTime::createFromFormat('d/m/Y', $end_date);
    if (!$date_obj) return '';
    
    // Format using WordPress date format setting
    $formatted = $date_obj->format(get_option('date_format'));
    
    if ($echo) {
        echo esc_html($formatted);
    }
    return $formatted;
}

/**
 * Display or get the event time
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The formatted time
 */
function vizou_event_time($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $time = get_field('event_time', $post_id);
    
    if (!$time) return '';
    
    if ($echo) {
        echo esc_html($time);
    }
    return $time;
}

/**
 * Display or get the venue name with optional link
 *
 * @param int $post_id Optional post ID
 * @param bool $link Whether to include the venue URL
 * @param bool $echo Whether to display or return
 * @return string The venue name with optional link
 */
function vizou_event_venue($post_id = null, $link = true, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $venue_name = get_the_title($post_id);
    $output = '';
    
    // Get venue URL if linking
    if ($link) {
        $venue_terms = get_the_terms($post_id, 'venue');
        $venue_url = '';
        
        if ($venue_terms && !is_wp_error($venue_terms)) {
            $venue_id = $venue_terms[0]->term_id;
            
            // Try with event_ prefix first, then without prefix
            $venue_url = get_term_meta($venue_id, 'event_venue_url', true);
            if (empty($venue_url)) {
                $venue_url = get_term_meta($venue_id, 'url', true);
            }
            
            if ($venue_url) {
                $output = '<a href="' . esc_url($venue_url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($venue_name) . '</a>';
            } else {
                $output = esc_html($venue_name);
            }
        }
    } else {
        $output = esc_html($venue_name);
    }
    
    if ($echo) {
        echo wp_kses_post($output);
    }
    return $output;
}

/**
 * Get venue URL
 *
 * @param int $post_id Optional post ID
 * @return string The venue URL or empty string
 */
function vizou_event_venue_url($post_id = null) {
    $post_id = $post_id ?: get_the_ID();
    $venue_terms = get_the_terms($post_id, 'venue');
    
    if (!$venue_terms || is_wp_error($venue_terms)) return '';
    
    $venue_id = $venue_terms[0]->term_id;
    
    // Try with event_ prefix first, then without prefix
    $venue_url = get_term_meta($venue_id, 'event_venue_url', true);
    if (empty($venue_url)) {
        $venue_url = get_term_meta($venue_id, 'url', true);
    }
    
    return $venue_url;
}

/**
 * Display or get the artist name with optional link
 *
 * @param int $post_id Optional post ID
 * @param bool $link Whether to include the artist URL
 * @param bool $echo Whether to display or return
 * @return string The artist name with optional link
 */
function vizou_event_artist($post_id = null, $link = true, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $output = '';
    
    $artist_terms = get_the_terms($post_id, 'artist');
    if (!$artist_terms || is_wp_error($artist_terms)) return '';
    
    $artist = $artist_terms[0];
    $artist_url = esc_url(get_field('event_artist_url', $post_id));
    
    if ($link && $artist_url) {
        $output = '<span class="event-artist"><a href="' . esc_attr($artist_url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($artist->name) . '</a></span>';
    } else {
        $output = '<span class="event-artist">' . esc_html($artist->name) . '</span>';
    }
    
    if ($echo) {
        echo wp_kses_post($output);
    }
    return $output;
}

/**
 * Get artist URL
 *
 * @param int $post_id Optional post ID
 * @return string The artist URL or empty string
 */
function vizou_event_artist_url($post_id = null) {
    $post_id = $post_id ?: get_the_ID();
    return esc_url(get_field('event_artist_url', $post_id));
}

/**
 * Display or get the venue address (street)
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The venue street address
 */
function vizou_event_venue_street($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $venue_terms = get_the_terms($post_id, 'venue');
    
    if (!$venue_terms || is_wp_error($venue_terms)) return '';
    
    $venue_id = $venue_terms[0]->term_id;
    
    // Try with event_ prefix first, then without prefix
    $street = get_term_meta($venue_id, 'event_street', true);
    if (empty($street)) {
        $street = get_term_meta($venue_id, 'street', true);
    }
    
    if ($echo) {
        echo esc_html($street);
    }
    return $street;
}

/**
 * Display or get the venue city
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The venue city
 */
function vizou_event_venue_city($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $venue_terms = get_the_terms($post_id, 'venue');
    
    if (!$venue_terms || is_wp_error($venue_terms)) return '';
    
    $venue_id = $venue_terms[0]->term_id;
    
    // Try with event_ prefix first, then without prefix
    $city = get_term_meta($venue_id, 'event_city', true);
    if (empty($city)) {
        $city = get_term_meta($venue_id, 'city', true);
    }
    
    if ($echo) {
        echo esc_html($city);
    }
    return $city;
}

/**
 * Display or get the venue province/state
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The venue province/state
 */
function vizou_event_venue_province($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $venue_terms = get_the_terms($post_id, 'venue');
    
    if (!$venue_terms || is_wp_error($venue_terms)) return '';
    
    $venue_id = $venue_terms[0]->term_id;
    
    // Try with event_ prefix first, then without prefix
    $province = get_term_meta($venue_id, 'event_province', true);
    if (empty($province)) {
        $province = get_term_meta($venue_id, 'province', true);
    }
    
    if ($echo) {
        echo esc_html($province);
    }
    return $province;
}

/**
 * Display or get the venue country
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The venue country
 */
function vizou_event_venue_country($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $venue_terms = get_the_terms($post_id, 'venue');
    
    if (!$venue_terms || is_wp_error($venue_terms)) return '';
    
    $venue_id = $venue_terms[0]->term_id;
    
    // Try with event_ prefix first, then without prefix
    $country = get_term_meta($venue_id, 'event_country', true);
    if (empty($country)) {
        $country = get_term_meta($venue_id, 'country', true);
    }
    
    if ($echo) {
        echo esc_html($country);
    }
    return $country;
}

/**
 * Display or get the event status
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The formatted event status
 */
function vizou_event_status($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $status = get_field('event_status', $post_id);
    
    if (!$status || strtolower(trim($status)) === 'active') return '';
    
    $formatted = Vizou_Events_Settings::get_status_label($status);
    
    if ($echo) {
        echo '<span class="event-status">' . esc_html($formatted) . '</span>';
    }
    return $formatted;
}

/**
 * Display or get the tickets link if available
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The tickets link HTML or empty string
 */
function vizou_event_ticket_button($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $ticket_url = get_field('event_ticket_url', $post_id);
    $event_status = get_field('event_status', $post_id);
    
    if (!$ticket_url || strtolower(trim($event_status)) !== 'active') return '';
    
    $label = Vizou_Events_Settings::get_option('label_tickets');
    $output = '<a class="event-tickets button" href="' . esc_url($ticket_url) . '" target="_blank" rel="noopener noreferrer">' . 
              esc_html($label) . '</a>';
    
    if ($echo) {
        echo wp_kses_post($output);
    }
    return $output;
}

/**
 * Display or get the event notes
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The event notes HTML
 */
function vizou_event_notes($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    $content = get_post_field('post_content', $post_id);
    
    if (empty($content)) return '';
    
    // Just get the raw content without the_content filter
    // This avoids automatic <p> tag wrapping
    $output = wp_kses_post($content);
    
    if ($echo) {
        echo wp_kses_post($output);
    }
    return $output;
}

/**
 * Get ticket URL
 *
 * @param int $post_id Optional post ID
 * @return string The ticket URL or empty string
 */
function vizou_event_ticket_url($post_id = null) {
    $post_id = $post_id ?: get_the_ID();
    return esc_url(get_field('event_ticket_url', $post_id));
}

/**
 * Display or get the tour name
 *
 * @param int $post_id Optional post ID
 * @param bool $echo Whether to display or return
 * @return string The tour name
 */
function vizou_event_tour($post_id = null, $echo = true) {
    $post_id = $post_id ?: get_the_ID();
    
    $tour_terms = get_the_terms($post_id, 'tour');
    if (!$tour_terms || is_wp_error($tour_terms)) return '';
    
    $tour = $tour_terms[0];
    $output = esc_html($tour->name);
    
    if ($echo) {
    echo esc_html($tour->name);
    }
    return esc_html($tour->name);
}

/**
 * Get event permalink
 *
 * @param int $post_id Optional post ID
 * @return string The event permalink
 */
function vizou_event_permalink($post_id = null) {
    if (!$post_id) {
        global $post;
        $post_id = $post ? $post->ID : get_the_ID();
    }
    
    if (!$post_id) return '';
    
    return get_permalink($post_id);
}

/**
 * Check if this is a past event
 *
 * @param int $post_id Optional post ID
 * @return bool True if this is a past event
 */
function vizou_is_past_event($post_id = null) {
    $post_id = $post_id ?: get_the_ID();
    $event_date = get_field('event_start', $post_id);
    
    if (!$event_date) return false;
    
    try {
        // First try d/m/Y format (ACF default)
        $event_datetime = DateTime::createFromFormat('d/m/Y', $event_date);
        if (!$event_datetime) {
            // If that fails, try Y-m-d format
            $event_datetime = DateTime::createFromFormat('Y-m-d', $event_date);
        }
        
        if ($event_datetime) {
            // Set time to start of day for consistent comparison
            $event_datetime->setTime(0, 0, 0);
            
            $current_datetime = new DateTime();
            $current_datetime->setTime(0, 0, 0);
            
            return ($event_datetime < $current_datetime);
        }
    } catch (Exception $e) {
        // Fail silently
    }
    
    return false;
}

/**
 * Check if this is a cancelled event
 *
 * @param int $post_id Optional post ID
 * @return bool True if this is a cancelled event
 */
function vizou_is_cancelled_event($post_id = null) {
    $post_id = $post_id ?: get_the_ID();
    $event_status = get_field('event_status', $post_id);
    
    if ($event_status) {
        return (strcasecmp(sanitize_text_field($event_status), 'Cancelled') === 0);
    }
    
    return false;
}

/**
 * Display or get the request show link if available
 *
 * @param bool $echo Whether to display or return
 * @return string The request show link HTML or empty string
 */
function vizou_request_show_link($echo = true) {
    $text = Vizou_Events_Settings::get_option('request_show_text');
    $url = Vizou_Events_Settings::get_option('request_show_url');
    
    // Only return content if URL exists
    if (empty($url)) {
        return '';
    }
    
    // Use default text if none provided
    if (empty($text)) {
        $text = esc_html__('Request a show', 'vizou-events-manager');
    }
    
    $output = '<a class="request-show-link button" href="' . esc_url($url) . '" target="_blank" rel="noopener noreferrer">' . 
              esc_html($text) . '</a>';
    
    if ($echo) {
        echo wp_kses_post($output);
    }
    return $output;
}