<?php
/**
 * Handle venue creation and management
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Vizou_Events_Venue_Handler {
    public static function register() {
        // Handle venue field attributes
        add_filter('acf/prepare_field', array(__CLASS__, 'add_venue_field_attributes'));
      
        // Handle venue creation and updates
        add_action('acf/save_post', array(__CLASS__, 'handle_venue_creation'), 20);
      
        // Add AJAX handlers
        add_action('wp_ajax_get_venue_details', array(__CLASS__, 'get_venue_details'));
        
        // Add scripts for venue handling
        add_action('acf/input/admin_footer', array(__CLASS__, 'add_venue_scripts'));
        
        // Update venue field choices
        add_filter('acf/load_field/name=event_name', array(__CLASS__, 'update_venue_field_choices'));
    }
    
    /**
     * Add data attributes to venue-related fields for JavaScript targeting
     */
    public static function add_venue_field_attributes($field) {
        $venue_fields = array(
            'event_street' => 'street',
            'event_city' => 'city', 
            'event_province' => 'province', 
            'event_country' => 'country', 
            'event_venue_url' => 'url'
        );

        if (isset($venue_fields[$field['name']])) {
            $field['wrapper']['data-venue-field'] = $venue_fields[$field['name']];
        }
        
        return $field;
    }
    
    /**
     * Handle venue creation when saving an event
     */
    public static function handle_venue_creation($post_id) {
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id) || get_post_type($post_id) !== 'event') {
            return;
        }

        $selected_venue = get_field('event_name', $post_id);

        $fields_to_update = array(
            'event_street',
            'event_city',
            'event_province',
            'event_country',
            'event_venue_url'
        );
        
        if (!$selected_venue) {
            $venue_name = get_the_title($post_id);
            $new_venue = wp_insert_term(sanitize_text_field($venue_name), 'venue');

            if (!is_wp_error($new_venue)) {
                $venue_id = absint($new_venue['term_id']);
                
                foreach ($fields_to_update as $field_name) {
                    $value = get_field($field_name, $post_id);
                    if ($value) {
                        // Store with event_ prefix for standardization
                        update_term_meta($venue_id, $field_name, sanitize_text_field($value));
                        
                        // Special case for URL - also store as 'url' for compatibility
                        if ($field_name === 'event_venue_url') {
                            update_term_meta($venue_id, 'url', sanitize_text_field($value));
                        }
                    }
                }

                wp_set_object_terms($post_id, $venue_id, 'venue');
                update_field('event_name', $venue_id, $post_id);
            }
        } elseif ($selected_venue) {
            $venue_id = absint(is_object($selected_venue) ? $selected_venue->term_id : $selected_venue);

            foreach ($fields_to_update as $field_name) {
                $value = get_field($field_name, $post_id);
                if ($value) {
                    // Store with event_ prefix for standardization
                    update_term_meta($venue_id, $field_name, sanitize_text_field($value));
                    
                    // Special case for URL - also store as 'url' for compatibility
                    if ($field_name === 'event_venue_url') {
                        update_term_meta($venue_id, 'url', sanitize_text_field($value));
                    }
                    
                    // Clean up old format if it exists (without event_ prefix)
                    $old_key = str_replace('event_', '', $field_name);
                    if ($old_key !== $field_name) { // Avoid deleting 'url' for venue_url
                        delete_term_meta($venue_id, $old_key);
                    }
                }
            }
        }
    }
    
    /**
     * AJAX handler for getting venue details
     */
    public static function get_venue_details() {
         if (!current_user_can('edit_posts')) {
            wp_send_json_error('Unauthorized access');
        }
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'get_venue_details')) {
            wp_send_json_error('Invalid security token');
        }

        // Check for term_id
        if (!isset($_POST['term_id'])) {
            wp_send_json_error('No term ID provided');
        }

        $term_id = absint($_POST['term_id']);
        
        $details = array(
            'street' => '',
            'city' => '',
            'province' => '', 
            'country' => '',
            'url' => ''
        );
        
        // Try to get data with event_ prefix first (standardized format)
        $details['street'] = get_term_meta($term_id, 'event_street', true);
        $details['city'] = get_term_meta($term_id, 'event_city', true);
        $details['province'] = get_term_meta($term_id, 'event_province', true);
        $details['country'] = get_term_meta($term_id, 'event_country', true);
        $details['url'] = get_term_meta($term_id, 'event_venue_url', true);
        
        // Fall back to non-prefixed format for any empty values
        if (empty($details['street'])) {
            $details['street'] = get_term_meta($term_id, 'street', true);
        }
        if (empty($details['city'])) {
            $details['city'] = get_term_meta($term_id, 'city', true);
        }
        if (empty($details['province'])) {
            $details['province'] = get_term_meta($term_id, 'province', true);
        }
        if (empty($details['country'])) {
            $details['country'] = get_term_meta($term_id, 'country', true);
        }
        if (empty($details['url'])) {
            $details['url'] = get_term_meta($term_id, 'url', true);
        }

        wp_send_json_success($details);
    }

    /**
     * Add scripts for venue field handling
     */
    public static function add_venue_scripts() {
    global $post_type;
        if ($post_type !== 'event') {
            return;
        }
        ?>
        <script type="text/javascript">
        (function($) {
            // Wait for both document and ACF to be ready
            $(document).ready(function() {
                initializeVenueFields();                  
            });

            function loadVenueDetails(termId) {
                if (!termId) return;
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_venue_details',
                        term_id: termId,
                        nonce: '<?php echo esc_js(wp_create_nonce("get_venue_details")); ?>'
                    },
                    success: function(response) {
                        if (response.success && response.data) {
                            const fieldMap = {
                                'street': 'event_street',
                                'city': 'event_city',
                                'province': 'event_province',
                                'country': 'event_country',
                                'url': 'event_venue_url'
                            };

                            Object.entries(response.data).forEach(([field, value]) => {
                                if (value) {
                                    const mappedField = fieldMap[field];
                                    const $input = $(`[data-name="${mappedField}"] input`);
                                    
                                    if ($input.length) {
                                        $input.val(value);
                                    }
                                }
                            });
                        }
                    }
                });
            }

            function initializeVenueFields() {
                var $venueSelect = $('[data-name="event_name"] select');
                var $titleField = $('#title');

                // Load initial venue details if venue is selected
                var initialVenue = $venueSelect.val();
                if (initialVenue) {
                    loadVenueDetails(initialVenue);
                }

                // Hide title prompt if title has value
                if ($titleField.val()) {
                    $('#title-prompt-text').hide();
                }

                // ONE-WAY SYNC: Venue selection → Title
                // Only populate title if user hasn't manually edited it
                $venueSelect.on('change', function() {
                    var selectedValue = $(this).val();
                    var selectedText = $(this).find('option:selected').text();
                    
                    // Update title from venue selection if title hasn't been manually edited
                    if (selectedText && !$titleField.data('user-edited')) {
                        $titleField.val(selectedText);
                        $('#title-prompt-text').hide();
                    }
                    
                    // Load venue details
                    if (selectedValue) {
                        loadVenueDetails(selectedValue);
                    }
                });

                // Mark title as manually edited if user types in it AFTER venue is selected
                // This prevents future venue changes from overwriting the custom title
                $titleField.on('input', function() {
                    var titleValue = $(this).val();
                    
                    // Show/hide prompt text
                    if (titleValue) {
                        $('#title-prompt-text').hide();
                    } else {
                        $('#title-prompt-text').show();
                    }

                    // If venue is already selected and user is editing title, mark as user-edited
                    if ($venueSelect.val()) {
                        $titleField.data('user-edited', true);
                    }
                });
            }
        })(jQuery);
        </script>
        <?php
    }
    
    /**
     * Update venue field choices in ACF field
     */
    public static function update_venue_field_choices($field) {
        $field['choices'] = array();
        
        $venues = get_terms(array(
            'taxonomy' => 'venue',
            'hide_empty' => false,
        ));
        
        if (!empty($venues) && !is_wp_error($venues)) {
            foreach ($venues as $venue) {
                $field['choices'][$venue->term_id] = esc_html($venue->name);
            }
        }
        
        return $field;
    }
}

// Initialize venue handler
add_action('init', array('Vizou_Events_Venue_Handler', 'register'));