<?php
/**
 * Template: Event List Template
 *
 * This template renders each individual event in the events list.
 * To customize, copy this file to your-theme/vizou-events/vizou-events-list.php
 *
 * Available template tags:
 * - vizou_event_date() - Displays the event's start date
 * - vizou_event_end_date() - Displays the event's end date (for multi-day events)
 * - vizou_event_time() - Displays the event's time
 * - vizou_event_artist() - Displays the event's artist with optional link
 * - vizou_event_venue() - Displays the event's venue with optional link
 * - vizou_event_venue_street() - Displays the venue street
 * - vizou_event_venue_city() - Displays the venue city
 * - vizou_event_venue_province() - Displays the venue province
 * - vizou_event_venue_country() - Displays the venue country
 * - vizou_event_notes() - Displays the event description/notes
 * - vizou_event_status() - Displays the event status (if not "Active")
 * - vizou_event_ticket_button() - Displays the tickets button with link
 * - vizou_event_ticket_url() - Displays the tickets button with link
 * - vizou_event_tour() - Displays tour name
 *
 * @package Vizou_Events_Manager
 * @since 0.1.12
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;
?>

<!-- This is the default events list template -->
 
<!-- BEGIN EVENT -->
<?php
// Force proper class detection directly in this template
$classes = array('event');

// Directly check if this is a past event
$event_date = get_field('event_start');
if ($event_date) {
    try {
        // First try d/m/Y format (ACF default)
        $event_datetime = DateTime::createFromFormat('d/m/Y', $event_date);
        if (!$event_datetime) {
            // If that fails, try Y-m-d format
            $event_datetime = DateTime::createFromFormat('Y-m-d', $event_date);
        }
        
        if ($event_datetime) {
            // Set time to end of day for consistent comparison
            $event_datetime->setTime(0, 0, 0);
            
            $current_datetime = new DateTime();
            $current_datetime->setTime(0, 0, 0);
            
            if ($event_datetime < $current_datetime) {
                $classes[] = 'event-past';
            }
        }
    } catch (Exception $e) {
        // Fail silently
    }
}

// Check if this is a cancelled event
$event_status = get_field('event_status');
if ($event_status && strcasecmp(trim($event_status), 'cancelled') === 0) {
    $classes[] = 'event-cancelled';
}
?>

<div class="<?php echo esc_attr(implode(' ', array_map('sanitize_html_class', $classes))); ?>">

    <div class="left">

        <!-- DATE, TIME -->
        <p class="event-date">
            <?php 
            // Get the raw date value
            $post_id = get_the_ID();
            $start_date_raw = get_field('event_start', $post_id);
            
            if ($start_date_raw) {
                // First create DateTime object from stored d/m/Y format
                $date_obj = DateTime::createFromFormat('d/m/Y', $start_date_raw);
                if ($date_obj) {
                    // Use format() instead of getTimestamp() + wp_date()
                    echo '<span class="event-start-date">' . esc_html($date_obj->format(get_option('date_format'))) . '</span>';
                }
            }
        
            // Handle end date if present
            $end_date_raw = get_field('event_end', $post_id);
            if (!empty($end_date_raw)) {
                $end_date_obj = DateTime::createFromFormat('d/m/Y', $end_date_raw);
                if ($end_date_obj) {
                    echo '<span class="event-end-date">-' . esc_html($end_date_obj->format(get_option('date_format'))) . '</span>';
                }
            }
            ?>

            <?php 
            // Add time, but only if it exists
            $time = get_field('event_time', $post_id);
            if (!empty($time)): 
            ?>
                <span class="event-time"> - <?php echo esc_html($time); ?></span>
            <?php endif; ?>
        </p>

       <!-- TITLE (VENUE, ARTIST) -->
        <p class="event-title">
            <?php 
            // Display format depends on the setting
            $display_format = Vizou_Events_Settings::get_option('display_format');
            
            // Check if artist exists
            $has_artist = false;
            $artist_terms = get_the_terms(get_the_ID(), 'artist');
            if (!empty($artist_terms) && !is_wp_error($artist_terms)) {
                $has_artist = true;
            }
            
            if ($display_format === 'artist_first') {
                // Artist first
                if ($has_artist) {
                    echo '<span class="event-artist artist-first">';
                    vizou_event_artist();
                    echo '</span>';
                    echo '<span class="event-venue">';
                    vizou_event_venue();
                    echo '</span>';
                } else {
                    // Only venue if no artist
                    echo '<span class="event-venue">';
                    vizou_event_venue();
                    echo '</span>';
                }
            } else {
                    // Venue first (default)
                    echo '<span class="event-venue venue-first">';
                    vizou_event_venue();
                    echo '</span>';
                if ($has_artist) {
                    echo '<span class="event-artist">';
                    vizou_event_artist();
                    echo '</span>';
                }
            }
            ?>
        </p>

        <!-- LOCATION -->
        <p class="event-details">

            <span class="event-address">
                <?php if (vizou_event_venue_street(null, false)): ?>
                    <span class="event-street"><?php vizou_event_venue_street(); ?>, </span>
                <?php endif; ?>
                
                <?php if (vizou_event_venue_city(null, false)): ?>
                    <span class="event-city"><?php vizou_event_venue_city(); ?>, </span>
                <?php endif; ?>
                
                <?php if (vizou_event_venue_province(null, false)): ?>
                    <span class="event-province"><?php vizou_event_venue_province(); ?></span>
                <?php endif; ?>
                
                <?php if (vizou_event_venue_country(null, false)): ?>
                    <span class="event-country"> (<?php vizou_event_venue_country(); ?>)</span>
                <?php endif; ?>
            </span> 

            <!-- NOTES -->
            <?php if (vizou_event_notes(null, false)): ?>
                <span class="event-notes">
                    <?php vizou_event_notes(); ?>
                </span>
            <?php endif; ?>

            <!-- READ MORE LINK - Only if content in the editor -->
            <?php if (get_the_content()): ?>
                <a href="<?php echo esc_url(vizou_event_permalink()); ?>" title="Info" class="event-info-link">Info</a>
            <?php endif; ?>

            <!-- EDIT EVENT -->
            <?php edit_post_link(esc_html__('Edit', 'vizou-events-manager'), '', ''); ?>

        </p>

    </div>

    <div class="right">

        <!-- THUMBNAIL -->
        <?php if (has_post_thumbnail()): ?>
            <?php the_post_thumbnail('large', array(
                'class' => 'event-thumb',
                'loading' => 'lazy',
                'alt' => get_the_title()
            )); ?>
        <?php endif; ?>

        <!-- TICKETS OR STATUS -->
        <p class="event-tickets">
            <?php 
            // Get event status
            $event_status = get_field('event_status');
            $ticket_url = get_field('event_ticket_url');
            
            // Show status badge if not active, otherwise show ticket button if URL exists
            if ($event_status && strtolower(trim($event_status)) !== 'active') {
                // Display status
                $status_label = Vizou_Events_Settings::get_status_label($event_status);
                echo '<span class="event-status button">' . esc_html($status_label) . '</span>';
            } elseif ($ticket_url) {
                // Display ticket button
                vizou_event_ticket_button();
            }
            ?>
        </p>

    </div>
</div>