<?php
/**
 * Plugin Name: Vizou Events Manager
 * Plugin URI: https://explicable.ca
 * Description: Events management for artists. Requires Advanced Custom Fields (free or Pro) plugin to be active.
 * Version: 0.1.13
 * Requires at least: 6.0
 * Tested up to: 6.8
 * Requires PHP: 8.0
 * Author: Dana Whittle
 * Author URI: https://vizou.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: vizou-events-manager
 */

// Prevent direct access
defined('ABSPATH') || exit;

class Vizou_Events_Manager {
   private static $instance = null;
   private $plugin_path;
   
   private function __construct() {
       $this->plugin_path = plugin_dir_path(__FILE__);
       $this->init();
   }
   
   public static function get_instance() {
       if (null === self::$instance) {
           self::$instance = new self();
       }
       return self::$instance;
   }
   
   private function init() {
       // Check if ACF is active
       add_action('admin_notices', [$this, 'check_acf']);
       
       // Only initialize if ACF is active
       if (class_exists('ACF')) {
           $this->include_files();
           $this->setup_hooks();
       }
   }
   
   public function check_acf() {
       if (!class_exists('ACF')) {
           ?>
           <div class="notice notice-error">
               <p><?php esc_html_e('Vizou Events Manager requires Advanced Custom Fields (free version) to be installed and activated.', 'vizou-events-manager'); ?></p>
           </div>
           <?php
       }
   }
   
   private function include_files() {
       $required_files = [
            'includes/admin-columns.php',
            'includes/artist-handler.php',
            'includes/field-groups.php',
            'includes/help.php',
            'includes/import-export.php',
            'includes/permalink-handler.php',
            'includes/post-types.php',
            'includes/settings.php',
            'includes/shortcodes.php',
            'includes/taxonomies.php',
            'includes/template-tags.php',
            'includes/venue-handler.php'
       ];

       foreach ($required_files as $file) {
           $full_path = $this->plugin_path . $file;
           if (file_exists($full_path)) {
               require_once $full_path;
           }
       }
   }
   
   private function setup_hooks() {
       // Initialize components on appropriate hooks
       add_action('init', [$this, 'init_post_types'], 0);
       add_action('init', [$this, 'init_taxonomies'], 0);
       add_action('acf/init', [$this, 'init_field_groups']);
       add_action('init', [$this, 'init_shortcodes']);
       add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
       
       // Verify classes exist before registering
       if (class_exists('Vizou_Events_Settings')) {
           Vizou_Events_Settings::register();
       }
       if (class_exists('Vizou_Events_Import_Export')) {
           Vizou_Events_Import_Export::register();
       }
       if (class_exists('Vizou_Events_Help')) {
           Vizou_Events_Help::register();
       }
   }
   
   public function init_post_types() {
       if (class_exists('Vizou_Events_Post_Types')) {
           Vizou_Events_Post_Types::register();
       }
   }
   
   public function init_taxonomies() {
       if (class_exists('Vizou_Events_Taxonomies')) {
           Vizou_Events_Taxonomies::register();
       }
   }
   
   public function init_field_groups() {
        if (class_exists('Vizou_Events_Field_Groups')) {
            Vizou_Events_Field_Groups::register();
        }

        // Start date field filter
        add_filter('acf/load_field/key=field_5cf91f512ad24', function($field) {
            $field['display_format'] = sanitize_text_field(get_option('date_format'));
            $field['return_format'] = 'd/m/Y';  // Consistent storage format
            return $field;
        });
        
        // End date field filter
        add_filter('acf/load_field/key=field_63e64f152b866', function($field) {
            $field['display_format'] = sanitize_text_field(get_option('date_format'));
            $field['return_format'] = 'd/m/Y';  // Consistent storage format
            return $field;
        });
        
        // Enhanced time format filter with sanitization
        add_filter('acf/load_field/key=field_5cf91f89677cc', function($field) {
            $wp_time_format = sanitize_text_field(get_option('time_format'));
            
            // Determine if we're using 12 or 24 hour format
            $is_24_hour = (strpos($wp_time_format, 'H') !== false || strpos($wp_time_format, 'G') !== false);
            
            // Set display format based on site setting
            $field['display_format'] = $is_24_hour ? 'H:i' : 'g:i a';
            
            // Match return format to display format
            $field['return_format'] = $field['display_format'];
            
            // Ensure picker is enabled
            $field['allow_null'] = 0;
            $field['show_time_select'] = 1;
            
            // Handle different format display if needed
            if (strpos($wp_time_format, '\h') !== false || strpos($wp_time_format, 'H\h') !== false) {
                add_filter('acf/format_value/key=' . sanitize_key($field['key']), function($value) {
                    if (!empty($value)) {
                        $timestamp = strtotime($value);
                        return $timestamp ? wp_kses_post(date('H\hi', $timestamp)) : '';
                    }
                    return $value;
                });
            }
            
            return $field;
        });
        
        // Add filter for the status field to sync with custom labels
        add_filter('acf/load_field/key=field_5fa6b27b6dd89', function($field) {
            // Reset choices array
            $field['choices'] = array();
            
            // The key is what will be stored in the database
            // The value is what will be displayed to the user
            $field['choices']['Active'] = esc_html(Vizou_Events_Settings::get_status_label('active'));
            $field['choices']['Cancelled'] = esc_html(Vizou_Events_Settings::get_status_label('cancelled'));
            $field['choices']['Sold out'] = esc_html(Vizou_Events_Settings::get_status_label('sold_out'));
            $field['choices']['Reschedule'] = esc_html(Vizou_Events_Settings::get_status_label('reschedule'));
            
            return $field;
        });
    }
   
    public function activate() {
       // Verify current user has permission to activate plugins
       if (!current_user_can('activate_plugins')) {
           return;
       }
       flush_rewrite_rules();
   }
   
   public function deactivate() {
       // Verify current user has permission to deactivate plugins
       if (!current_user_can('activate_plugins')) {
           return;
       }
       flush_rewrite_rules();
   }

   public function get_template_path($template_name) {
       $template_name = sanitize_file_name($template_name);
       $theme_template = locate_template("vizou-events/{$template_name}");
       
       if ($theme_template && validate_file($theme_template) === 0) {
           return $theme_template;
       }
       
       $plugin_template = $this->plugin_path . "templates/{$template_name}";
       if (validate_file($plugin_template) === 0) {
           return $plugin_template;
       }
       
       return '';
   }

   public function init_shortcodes() {
       if (class_exists('Vizou_Events_Shortcodes')) {
           Vizou_Events_Shortcodes::register();
       }
   }

   public function enqueue_assets() {
       // Verify the option exists and is a boolean
       $disable_css = (bool) Vizou_Events_Settings::get_option('disable_default_css');
       
       if (!$disable_css) {
           wp_register_style(
               'vizou-events-manager',
               esc_url(plugins_url('assets/css/vizou-events.css', __FILE__)),
               array(),
               '0.1.1'
           );
           wp_enqueue_style('vizou-events-manager');
       }
   }
}

// Initialize plugin
function vizou_events_manager() {
   return Vizou_Events_Manager::get_instance();
}

// Register activation/deactivation hooks
register_activation_hook(__FILE__, [vizou_events_manager(), 'activate']);
register_deactivation_hook(__FILE__, [vizou_events_manager(), 'deactivate']);

// Kick off the plugin
add_action('plugins_loaded', 'vizou_events_manager');
