<?php
/**
 * Plugin Name: Vizou Quickloop
 * Plugin URI: https://explicable.ca
 * Description: A simple shortcode to create custom post loops inside pages.
 * Version: 1.0.8
 * Requires at least: 6.0
 * Tested up to: 6.8
 * Requires PHP: 8.0
 * Author: Dana Whittle
 * Author URI: https://vizou.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: vizou-quickloop
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

function vizou_custom_quickloop($atts) {
    // Parse attributes
    $atts = shortcode_atts(array(
        'posts_per_page' => '',
        'p' => '',
        'tag' => '',
        'orderby' => 'menu_order',
        'order' => 'asc',
        'offset' => '',
        'cat' => '',
        'class' => '',
        'exclude' => '',
        'include' => '',
        'show_title' => '',
        'show_excerpt' => '',
        'show_thumbnail' => '',
        'show_date' => '',
        'show_permalink' => ''
    ), $atts);

    // Build query args
    $query_args = array();
    
    if ($atts['posts_per_page'] !== '') {
        $query_args['posts_per_page'] = intval($atts['posts_per_page']);
    }
    if (!empty($atts['cat'])) {
        $query_args['cat'] = absint($atts['cat']);
    }
    if (!empty($atts['p'])) {
        $query_args['p'] = absint($atts['p']);
    }
    if (!empty($atts['orderby'])) {
        $query_args['orderby'] = sanitize_key($atts['orderby']);
    }
    if (!empty($atts['order'])) {
        $query_args['order'] = sanitize_key($atts['order']);
    }
    if (!empty($atts['tag'])) {
        $query_args['tag'] = sanitize_text_field($atts['tag']);
    }
    
    // Handle offset
    if ($atts['offset'] !== '') {
        if ($atts['posts_per_page'] === '-1') {
            $count_query = new WP_Query(array_merge($query_args, array('posts_per_page' => -1, 'fields' => 'ids')));
            $total_posts = $count_query->post_count;
            wp_reset_postdata();
            
            $query_args['posts_per_page'] = $total_posts - intval($atts['offset']);
            $query_args['offset'] = intval($atts['offset']);
        } else {
            $query_args['offset'] = intval($atts['offset']);
        }
    }

    // Parse display options
    $display_options = vizou_parse_display_options($atts);

    // Performance optimizations for the main query
    $query_args['no_found_rows'] = true; // Don't calculate total rows unless pagination is needed
    $query_args['update_post_meta_cache'] = false; // Don't update meta cache unless needed
    $query_args['update_post_term_cache'] = false; // Don't update term cache unless needed

    // Start output buffering
    ob_start();
    
    // Run the query with a unique name to prevent conflicts
    $vizou_loop = new WP_Query($query_args);
    
    if ($vizou_loop->have_posts()) :
        // Check for template override with class-specific template hierarchy
        $template = vizou_get_template_path($atts['class']);
        
        if ($template && file_exists($template) && is_readable($template)) {
            // Create a local scope for the template to prevent variable conflicts
            (function() use ($vizou_loop, $atts, $template, $display_options) {
                include($template);
            })();
        } else {
        ?>
            <!-- START LAYOUT -->
            <div class="posts vizou-quickloop<?php echo !empty($atts['class']) ? ' ' . esc_attr($atts['class']) : ''; ?>">
                <?php while ($vizou_loop->have_posts()) : $vizou_loop->the_post(); ?>
                    <article id="post-<?php the_ID(); ?>" class="post">
                        <?php if ($display_options['thumbnail']) : ?>
                            <a href="<?php echo esc_url(get_permalink()); ?>" title="<?php echo esc_attr(get_the_title()); ?>">
                                <?php the_post_thumbnail('large'); ?>
                            </a>
                        <?php endif; ?>
                        <span class="post-content">
                            <?php if ($display_options['title']) : ?>
                                <h3>
                                    <a href="<?php echo esc_url(get_permalink()); ?>" title="<?php echo esc_attr(get_the_title()); ?>"><?php echo esc_html(get_the_title()); ?></a>
                                </h3>
                            <?php endif; ?>
                            <?php if ($display_options['date']) : ?>
                                <time class="post-date" datetime="<?php echo esc_attr(get_the_date('c')); ?>"><?php echo esc_html(get_the_date()); ?></time>
                            <?php endif; ?>
                            <?php if ($display_options['excerpt']) : ?>
                                <?php echo esc_html(get_the_excerpt()); ?>
                            <?php endif; ?>
                        </span>
                        <?php if ($display_options['permalink']) : ?>
                            <a href="<?php echo esc_url(get_permalink()); ?>" class="read-more" title="Read more"></a>
                        <?php endif; ?>
                    </article>         
                <?php endwhile; ?>        
            </div>
            <!-- END LAYOUT -->
        <?php
        }
    endif;
    
    // Store the output before resetting post data
    $output = ob_get_clean();
    
    // Always reset the post data after the query
    wp_reset_postdata();
    
    return $output;
}

/**
 * Get the appropriate template path based on class hierarchy
 * 
 * @param string $class_name The class name from shortcode attributes
 * @return string|null Path to template file or null if none found
 */
function vizou_get_template_path($class_name = '') {
    $theme_dir = trailingslashit(get_template_directory());
    
    // If a class is specified, check for class-specific template first
    if (!empty($class_name)) {
        // Extract the first class from space-separated classes
        $classes = explode(' ', trim($class_name));
        $first_class = trim($classes[0]);
        
        if (!empty($first_class)) {
            // Sanitize first class for filename - remove spaces, convert to lowercase, allow hyphens
            $sanitized_class = sanitize_file_name(strtolower($first_class));
            $sanitized_class = preg_replace('/[^a-z0-9\-]/', '', $sanitized_class);
            
            if (!empty($sanitized_class)) {
                $class_template = $theme_dir . 'vizou-quickloop-template-' . $sanitized_class . '.php';
                if (file_exists($class_template) && is_readable($class_template)) {
                    return $class_template;
                }
            }
        }
    }
    
    // Fall back to general template
    $general_template = $theme_dir . 'vizou-quickloop-template.php';
    if (file_exists($general_template) && is_readable($general_template)) {
        return $general_template;
    }
    
    // No template override found
    return null;
}

/**
 * Parse display options for post elements
 * 
 * @param array $atts Shortcode attributes
 * @return array Display options with boolean values
 */
function vizou_parse_display_options($atts) {
    // Validate input
    if (!is_array($atts)) {
        $atts = array();
    }
    
    // Default display options - show everything that's in the template
    $defaults = array(
        'title' => true,
        'excerpt' => true,
        'thumbnail' => true,
        'date' => true,
        'permalink' => true
    );
    
    // Start with defaults
    $display_options = $defaults;
    
    // Handle exclude parameter (takes precedence)
    if (!empty($atts['exclude'])) {
        $excluded = array_map('trim', explode(',', sanitize_text_field($atts['exclude'])));
        foreach ($excluded as $element) {
            $element = sanitize_key($element);
            if (array_key_exists($element, $display_options)) {
                $display_options[$element] = false;
            }
        }
    }
    // Handle include parameter (exclusive - only show specified elements)
    elseif (!empty($atts['include'])) {
        // First, set all to false
        foreach ($display_options as $key => $value) {
            $display_options[$key] = false;
        }
        // Then enable only included elements
        $included = array_map('trim', explode(',', sanitize_text_field($atts['include'])));
        foreach ($included as $element) {
            $element = sanitize_key($element);
            if (array_key_exists($element, $display_options)) {
                $display_options[$element] = true;
            }
        }
    }
    
    return $display_options;
}

add_shortcode('vizou-quickloop', 'vizou_custom_quickloop');

/**
 * Add admin menu
 */
function vizou_quickloop_admin_menu() {
    add_options_page(
        'Vizou Quickloop',
        'Vizou Quickloop',
        'manage_options',
        'vizou-quickloop',
        'vizou_quickloop_admin_page'
    );
}
add_action('admin_menu', 'vizou_quickloop_admin_menu');

/**
 * Admin page content
 */
function vizou_quickloop_admin_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <p>A simple shortcode to create custom post loops inside pages with flexible display options and class-specific templates.</p>
        
        <div class="vizou-admin-tabs">
            <nav class="nav-tab-wrapper">
                <a href="#examples" class="nav-tab nav-tab-active">Examples</a>
                <a href="#parameters" class="nav-tab">Parameters</a>
                <a href="#elements" class="nav-tab">Display Elements</a>
                <a href="#template" class="nav-tab">Template Override</a>
                <a href="#class-templates" class="nav-tab">Class Templates</a>
            </nav>
            
            <div id="examples" class="tab-content active">
                <h2>Usage Examples</h2>
                
                <div class="vizou-example">
                    <h3>Basic Usage</h3>
                    <p>Get 2 posts from category 2, ordered by menu order:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="2" posts_per_page="2" orderby="menu_order" order="asc"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="2" posts_per_page="2" orderby="menu_order" order="asc"]'>Copy</button>
                    </div>
                </div>
                
                <div class="vizou-example">
                    <h3>Single Post</h3>
                    <p>Display a specific post by ID:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop p="35"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop p="35"]'>Copy</button>
                    </div>
                </div>
                
                <div class="vizou-example">
                    <h3>With Offset</h3>
                    <p>Get 5 posts from category 2, skip the first one:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="2" posts_per_page="5" offset="1"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="2" posts_per_page="5" offset="1"]'>Copy</button>
                    </div>
                </div>
                
                <div class="vizou-example">
                    <h3>Custom CSS Class</h3>
                    <p>Add a custom class to the container:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="9" posts_per_page="1" class="featured"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="9" posts_per_page="1" class="featured"]'>Copy</button>
                    </div>
                </div>
                
                <h3>Display Control Examples</h3>
                
                <div class="vizou-example">
                    <h3>Hide Elements</h3>
                    <p>Hide excerpt and date from all posts:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="1" posts_per_page="-1" exclude="excerpt,date"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="1" posts_per_page="-1" exclude="excerpt,date"]'>Copy</button>
                    </div>
                </div>
                
                <div class="vizou-example">
                    <h3>Show Only Specific Elements</h3>
                    <p>Show only title and thumbnail:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="2" include="title,thumbnail"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="2" include="title,thumbnail"]'>Copy</button>
                    </div>
                </div>
                
                <div class="vizou-example">
                    <h3>Class-Specific Templates</h3>
                    <p>Use custom templates for different layouts:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="5" class="gallery"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="5" class="gallery"]'>Copy</button>
                    </div>
                    <p><small>This will use <code>vizou-quickloop-template-gallery.php</code> if it exists in your theme.</small></p>
                </div>
                
                <div class="vizou-example">
                    <h3>Featured Posts Layout</h3>
                    <p>Create a special layout for featured posts:</p>
                    <div class="vizou-code">
                        <code>[vizou-quickloop cat="1" posts_per_page="3" class="featured-hero"]</code>
                        <button class="button vizou-copy" data-clipboard-text='[vizou-quickloop cat="1" posts_per_page="3" class="featured-hero"]'>Copy</button>
                    </div>
                    <p><small>This will use <code>vizou-quickloop-template-featured-hero.php</code> if it exists.</small></p>
                </div>
            </div>
            
            <div id="parameters" class="tab-content">
                <h2>All Parameters</h2>
                
                <h3>Query Parameters</h3>
                <table class="widefat">
                    <thead>
                        <tr>
                            <th>Parameter</th>
                            <th>Description</th>
                            <th>Example</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>cat</code></td>
                            <td>Category ID to filter posts</td>
                            <td>cat="2"</td>
                        </tr>
                        <tr>
                            <td><code>p</code></td>
                            <td>Specific post ID</td>
                            <td>p="35"</td>
                        </tr>
                        <tr>
                            <td><code>tag</code></td>
                            <td>Tag slug</td>
                            <td>tag="news"</td>
                        </tr>
                        <tr>
                            <td><code>posts_per_page</code></td>
                            <td>Number of posts (-1 for all)</td>
                            <td>posts_per_page="5"</td>
                        </tr>
                        <tr>
                            <td><code>orderby</code></td>
                            <td>Sort field (menu_order, date, title, etc.)</td>
                            <td>orderby="date"</td>
                        </tr>
                        <tr>
                            <td><code>order</code></td>
                            <td>Sort direction (asc, desc)</td>
                            <td>order="desc"</td>
                        </tr>
                        <tr>
                            <td><code>offset</code></td>
                            <td>Skip first X posts</td>
                            <td>offset="1"</td>
                        </tr>
                    </tbody>
                </table>
                
                <h3>Display Parameters</h3>
                <table class="widefat">
                    <thead>
                        <tr>
                            <th>Parameter</th>
                            <th>Description</th>
                            <th>Example</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>exclude</code></td>
                            <td>Comma-separated list of elements to hide</td>
                            <td>exclude="excerpt,date"</td>
                        </tr>
                        <tr>
                            <td><code>include</code></td>
                            <td>Comma-separated list of elements to show (exclusive)</td>
                            <td>include="title,thumbnail"</td>
                        </tr>
                        <tr>
                            <td><code>class</code></td>
                            <td>Custom CSS class for container + template selector</td>
                            <td>class="featured"</td>
                        </tr>
                    </tbody>
                </table>
                
                <div class="notice notice-info">
                    <p><strong>Note:</strong> The <code>exclude</code> parameter takes precedence over <code>include</code>. If neither is specified, all elements will display.</p>
                </div>
            </div>
            
            <div id="elements" class="tab-content">
                <h2>Display Elements</h2>
                <p>These are the post elements you can control with <code>exclude</code> and <code>include</code> parameters:</p>
                
                <table class="widefat">
                    <thead>
                        <tr>
                            <th>Element</th>
                            <th>Description</th>
                            <th>WordPress Function</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>title</code></td>
                            <td>Post title with permalink</td>
                            <td>get_the_title()</td>
                        </tr>
                        <tr>
                            <td><code>excerpt</code></td>
                            <td>Post excerpt text</td>
                            <td>get_the_excerpt()</td>
                        </tr>
                        <tr>
                            <td><code>thumbnail</code></td>
                            <td>Featured image with permalink</td>
                            <td>the_post_thumbnail()</td>
                        </tr>
                        <tr>
                            <td><code>date</code></td>
                            <td>Publication date</td>
                            <td>get_the_date()</td>
                        </tr>
                        <tr>
                            <td><code>permalink</code></td>
                            <td>Read more link</td>
                            <td>get_permalink()</td>
                        </tr>
                    </tbody>
                </table>
                
                <div class="notice notice-success">
                    <p><strong>Default Behavior:</strong> All elements show by default. Use <code>exclude</code> to hide specific elements or <code>include</code> to show only specific elements.</p>
                </div>
            </div>
            
            <div id="template" class="tab-content">
                <h2>Template Override</h2>
                <p>You can customize the HTML output by creating custom template files in your theme.</p>
                
                <h3>Template Hierarchy:</h3>
                <ol>
                    <li><code>vizou-quickloop-template-{class}.php</code> (class-specific template)</li>
                    <li><code>vizou-quickloop-template.php</code> (general template)</li>
                    <li>Built-in plugin template (default)</li>
                </ol>
                
                <h3>Creating a General Template:</h3>
                <ol>
                    <li>Copy the layout code from the plugin file (between START LAYOUT and END LAYOUT comments)</li>
                    <li>Create a new file called <code>vizou-quickloop-template.php</code></li>
                    <li>Place it in your active theme's root directory</li>
                    <li>Customize the HTML structure as needed</li>
                </ol>
                
                <h3>Using Display Options in Templates</h3>
                <p>Your custom templates have access to the <code>$display_options</code> array:</p>
                
                <div class="vizou-code-block">
                    <pre><code>&lt;?php if ($display_options['title']) : ?&gt;
    &lt;h3&gt;&lt;?php the_title(); ?&gt;&lt;/h3&gt;
&lt;?php endif; ?&gt;

&lt;?php if ($display_options['thumbnail']) : ?&gt;
    &lt;?php the_post_thumbnail('large'); ?&gt;
&lt;?php endif; ?&gt;

&lt;?php if ($display_options['date']) : ?&gt;
    &lt;time&gt;&lt;?php the_date(); ?&gt;&lt;/time&gt;
&lt;?php endif; ?&gt;

&lt;?php if ($display_options['excerpt']) : ?&gt;
    &lt;?php the_excerpt(); ?&gt;
&lt;?php endif; ?&gt;</code></pre>
                </div>
                
                <h3>Available Variables</h3>
                <ul>
                    <li><code>$vizou_loop</code> - The WP_Query object</li>
                    <li><code>$atts</code> - Array of shortcode attributes</li>
                    <li><code>$display_options</code> - Array of display flags (title, excerpt, thumbnail, date, permalink)</li>
                </ul>
                
                <div class="notice notice-warning">
                    <p><strong>Important:</strong> Always use the conditional checks with <code>$display_options</code> to respect the user's exclude/include settings.</p>
                </div>
            </div>
            
            <div id="class-templates" class="tab-content">
                <h2>Class-Specific Templates</h2>
                <p>Create different layouts for different use cases by using class-specific templates.</p>
                
                <h3>How It Works</h3>
                <p>When you use the <code>class</code> parameter in your shortcode, the plugin will look for a matching template file:</p>
                
                <div class="vizou-example">
                    <h4>Example: Gallery Layout</h4>
                    <p>Shortcode: <code>[vizou-quickloop cat="5" class="gallery"]</code></p>
                    <p>Template file: <code>vizou-quickloop-template-gallery.php</code></p>
                    <p>This could create a grid layout showing only thumbnails.</p>
                </div>
                
                <div class="vizou-example">
                    <h4>Example: Featured Posts</h4>
                    <p>Shortcode: <code>[vizou-quickloop cat="1" class="featured-hero"]</code></p>
                    <p>Template file: <code>vizou-quickloop-template-featured-hero.php</code></p>
                    <p>This could create a large hero section with big images and titles.</p>
                </div>
                
                <div class="vizou-example">
                    <h4>Example: Sidebar Widget</h4>
                    <p>Shortcode: <code>[vizou-quickloop cat="2" class="sidebar-list"]</code></p>
                    <p>Template file: <code>vizou-quickloop-template-sidebar-list.php</code></p>
                    <p>This could create a compact list with just titles and dates.</p>
                </div>
                
                <h3>Template Naming Rules</h3>
                <ul>
                    <li>Start with <code>vizou-quickloop-template-</code></li>
                    <li>Add your class name (lowercase, hyphens only)</li>
                    <li>End with <code>.php</code></li>
                    <li>Place in your theme's root directory</li>
                </ul>
                
                <h3>Class Name Sanitization</h3>
                <p>Class names are automatically sanitized for filenames:</p>
                <ul>
                    <li><code>Featured Posts</code> becomes <code>featured-posts</code></li>
                    <li><code>Gallery_Grid</code> becomes <code>gallery-grid</code></li>
                    <li><code>News Sidebar</code> becomes <code>news-sidebar</code></li>
                </ul>
                
                <h3>Example Template Structure</h3>
                <p>Here's a sample gallery template (<code>vizou-quickloop-template-gallery.php</code>):</p>
                
                <div class="vizou-code-block">
                    <pre><code>&lt;div class="posts vizou-quickloop gallery-grid"&gt;
    &lt;?php while ($vizou_loop-&gt;have_posts()) : $vizou_loop-&gt;the_post(); ?&gt;
        &lt;div class="gallery-item"&gt;
            &lt;?php if ($display_options['thumbnail']) : ?&gt;
                &lt;a href="&lt;?php echo esc_url(get_permalink()); ?&gt;"&gt;
                    &lt;?php the_post_thumbnail('medium'); ?&gt;
                &lt;/a&gt;
            &lt;?php endif; ?&gt;
            &lt;?php if ($display_options['title']) : ?&gt;
                &lt;h4&gt;&lt;?php the_title(); ?&gt;&lt;/h4&gt;
            &lt;?php endif; ?&gt;
        &lt;/div&gt;
    &lt;?php endwhile; ?&gt;
&lt;/div&gt;</code></pre>
                </div>
                
                <div class="notice notice-info">
                    <p><strong>Fallback:</strong> If no class-specific template is found, the plugin will use the general <code>vizou-quickloop-template.php</code> template, or the built-in template if neither exists.</p>
                </div>
            </div>
        </div>
    </div>
    
    <style>
    .vizou-admin-tabs .nav-tab-wrapper {
        border-bottom: 1px solid #ccd0d4;
        margin-bottom: 20px;
    }
    .vizou-admin-tabs .tab-content {
        display: none;
    }
    .vizou-admin-tabs .tab-content.active {
        display: block;
    }
    .vizou-example {
        background: #f9f9f9;
        border: 1px solid #e1e1e1;
        padding: 15px;
        margin-bottom: 20px;
        border-radius: 4px;
    }
    .vizou-example h3, .vizou-example h4 {
        margin-top: 0;
        color: #1d2327;
    }
    .vizou-code {
        background: #2d3748;
        color: #e2e8f0;
        padding: 12px;
        border-radius: 4px;
        font-family: monospace;
        position: relative;
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-top: 10px;
    }
    .vizou-code code {
        background: none;
        color: inherit;
        padding: 0;
    }
    .vizou-copy {
        margin-left: 10px;
        font-size: 12px;
    }
    .vizou-code-block {
        background: #2d3748;
        color: #e2e8f0;
        padding: 15px;
        border-radius: 4px;
        margin: 15px 0;
    }
    .vizou-code-block pre {
        margin: 0;
        white-space: pre-wrap;
    }
    .vizou-code-block code {
        background: none;
        color: inherit;
        padding: 0;
    }
    .notice {
        margin: 15px 0;
    }
    </style>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Tab functionality
        const tabs = document.querySelectorAll('.nav-tab');
        const contents = document.querySelectorAll('.tab-content');
        
        tabs.forEach(tab => {
            tab.addEventListener('click', function(e) {
                e.preventDefault();
                
                // Remove active class from all tabs and contents
                tabs.forEach(t => t.classList.remove('nav-tab-active'));
                contents.forEach(c => c.classList.remove('active'));
                
                // Add active class to clicked tab
                this.classList.add('nav-tab-active');
                
                // Show corresponding content
                const target = this.getAttribute('href').substring(1);
                const targetElement = document.getElementById(target);
                if (targetElement) {
                    targetElement.classList.add('active');
                }
            });
        });
        
        // Copy functionality with better error handling
        const copyButtons = document.querySelectorAll('.vizou-copy');
        copyButtons.forEach(button => {
            button.addEventListener('click', function() {
                const text = this.getAttribute('data-clipboard-text');
                
                // Check if clipboard API is available
                if (navigator.clipboard && navigator.clipboard.writeText) {
                    navigator.clipboard.writeText(text).then(() => {
                        showCopySuccess(this);
                    }).catch((err) => {
                        console.warn('Clipboard write failed:', err);
                        fallbackCopyText(text, this);
                    });
                } else {
                    // Fallback for older browsers
                    fallbackCopyText(text, this);
                }
            });
        });
        
        function showCopySuccess(button) {
            const originalText = button.textContent;
            button.textContent = 'Copied!';
            button.style.background = '#00a32a';
            button.style.color = '#fff';
            setTimeout(() => {
                button.textContent = originalText;
                button.style.background = '';
                button.style.color = '';
            }, 2000);
        }
        
        function fallbackCopyText(text, button) {
            // Create a temporary textarea
            const textArea = document.createElement('textarea');
            textArea.value = text;
            textArea.style.position = 'fixed';
            textArea.style.left = '-999999px';
            textArea.style.top = '-999999px';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            
            try {
                document.execCommand('copy');
                showCopySuccess(button);
            } catch (err) {
                console.warn('Fallback copy failed:', err);
                button.textContent = 'Copy failed';
                setTimeout(() => {
                    button.textContent = 'Copy';
                }, 2000);
            }
            
            document.body.removeChild(textArea);
        }
    });
    </script>
    <?php
}

/**
 * Enqueue admin styles and scripts
 */
function vizou_quickloop_admin_enqueue($hook) {
    if ($hook !== 'settings_page_vizou-quickloop') {
        return;
    }
    
    // Admin styles are included inline in the admin page function
}
add_action('admin_enqueue_scripts', 'vizou_quickloop_admin_enqueue');